﻿#!/usr/bin/python
# File: reportDMClashes.py
# Author: Yosi Izaq, aka WraithLord
# Description: script for reporting conflicts in DOM-III mod files
# Usage: reportDMClashes <list of .dm files>
"""
It works like this
a. go to mods dir
b. run: python [path/to/file]reportDMClashes.py  on list of .dm files to check, ex:
[/cygdrive/c/games/dominions3/mods/mod_not_in_use:]$ python /cygdrive/c/work/scripts/games/reportDMClashes.py  *.dm  
  
Warns go to stdout and logs go to reportDMClashes.log (set to log when >= INFO).

Snippet of very long list of conflicts when running check on a bunch of old mods:

Warning: found conflict! ID 2755 appears 3 times.
At Dominions 3000 v0.75.dm:4829, as type: monster, name Ice Samurai Battle Suit
At Githzerai.dm:426, as type: monster, name Psion
At Shangri La.dm:862, as type: monster, name Dmag Dpon
Warning: found conflict! ID 2525 appears 2 times.
At AI Ogre Kingdoms.dm:1351, as type: monster, name Slavegiant
At Dominions 3000 v0.75.dm:8532, as type: monster, name Primordia Regina
Warning: found conflict! ID 2748 appears 2 times.
At Githzerai.dm:287, as type: monster, name Soryo
At Shangri La.dm:601, as type: monster, name Rta Pa
Warning: found conflict! ID 2749 appears 2 times.
At Githzerai.dm:304, as type: monster, name Zerain
At Shangri La.dm:640, as type: monster, name Guardian of the Hidden Valley
Warning: found conflict! ID 2750 appears 3 times.
At Dominions 3000 v0.75.dm:4729, as type: monster, name Aka-Oni
At Githzerai.dm:323, as type: monster, name Inzen
At Shangri La.dm:679, as type: monster, name Mi G<F6>
Warning: found conflict! ID 2751 appears 3 times.
At Dominions 3000 v0.75.dm:4773, as type: monster, name Ao-Oni
At Githzerai.dm:341, as type: monster, name Ninja
At Shangri La.dm:711, as type: monster, name Mi G<F6> Rngon Pa
Warning: found conflict! ID 705 appears 2 times.
At Dominions 3000 v0.75.dm:617, as type: weapon, name Lightning Cannon
At Githzerai.dm:58, as type: weapon, name Fists of the Grandmaster
Warning: found conflict! ID 2754 appears 2 times.
At Dominions 3000 v0.75.dm:4801, as type: monster, name Fire Samurai Battle Suit
At Githzerai.dm:410, as type: monster, name Majishan
Warning: found conflict! ID 707 appears 4 times.
At Dominions 3000 v0.75.dm:651, as type: weapon, name Needler
At Githzerai.dm:76, as type: weapon, name Stunning Fist
At WH191andCBM16.dm:1234, as type: weapon, name Orna
At WH19andCBM16.dm:1227, as type: weapon, name Orna
Warning: found conflict! ID 2756 appears 3 times.
At Dominions 3000 v0.75.dm:4856, as type: monster, name Lightning Samurai Battle Suit
At Githzerai.dm:444, as type: monster, name Guru
At Shangri La.dm:897, as type: monster, name Rta Dmag Go
Warning: found conflict! ID 2205 appears 2 times.
At MC_NI_ Indy_Pack.dm:605, as type: monster, name item
At Nephilim.dm:249, as type: monster, name Feral
Warning: found conflict! ID 2757 appears 3 times.
At Dominions 3000 v0.75.dm:4884, as type: monster, name Gokenin
At Githzerai.dm:462, as type: monster, name Senin
At Shangri La.dm:936, as type: monster, name Rje
Warning: found conflict! ID 2758 appears 3 times.
At Dominions 3000 v0.75.dm:4914, as type: monster, name Wraith Suit Shinobi
At Githzerai.dm:481, as type: monster, name Dai-Zerain
At Shangri La.dm:976, as type: monster, name Drapa
Warning: found conflict! ID 2759 appears 3 times.
At Dominions 3000 v0.75.dm:4946, as type: monster, name Hatamoto
At Githzerai.dm:513, as type: monster, name Ghost Gith Commoner
At Shangri La.dm:1018, as type: monster, name Ngagspa
Warning: found conflict! ID 2760 appears 3 times.
At Dominions 3000 v0.75.dm:4976, as type: monster, name Daimyo
At Githzerai.dm:536, as type: monster, name Seirei
At Shangri La.dm:1057, as type: monster, name Demon Hunter
Warning: found conflict! ID 2761 appears 3 times.
At Dominions 3000 v0.75.dm:5006, as type: monster, name Dai-Oni
At Githzerai.dm:552, as type: monster, name Tamashi
At Shangri La.dm:1096, as type: monster, name Ragyapa
...

"""

"""
GPL licensing

(C) Copyright, 2011 Yosi Izaq.

This file, reportDMClashes.py, is part of reportDMClashes programm.

reportDMClashes is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

reportDMClashes is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with reportDMClashes.  If not, see <http://www.gnu.org/licenses/>.
"""

"""
Top lvl design
1. Syntax of .dm
Example -

#newarmor 340
#name "Armor of Bugginess"
#def -1
#type 5
#enc 1
#prot 13
#rcost 11
#end

#newarmor 340
#name "Armor of Suckiness"
#def -3
#type 5
#enc 5
#prot 2
#rcost 15
#end

2. container class for item (contains it's ID and name + file name and line #)
 
a. process argv

b. treat each item as file name and load each item into a [id, item(id, name, file, line)] dictionary

 -- regexp defs for parsing a mod entry

c. data structure that will facilitate the end result. Meaning:

Mod_Items {} (dictionary), keys --------------- Item (ID, Type) ------------
						:
			item list [], item1, item2, ...
				item, class: ID, type, name, file, line, attrs		
Conflicts are by definition Mod_Item Keys that have values with more than one item

"""

import shutil, tarfile,ConfigParser, os, re, sys, glob, logging, copy, pickle

#Logging configuration:
FORMAT = "%(asctime)-15s %(name)s %(levelname)s %(message)s"

LEVELS = {'debug': logging.DEBUG,
          'info': logging.INFO,
          'warning': logging.WARNING,
          'error': logging.ERROR,
          'critical': logging.CRITICAL}


#if len(sys.argv) > 1:
#level_name = sys.argv[1]
#level_name = 'debug'
level_name = 'info'
level = LEVELS.get(level_name, logging.NOTSET)
logging.basicConfig(level=level, format=FORMAT, filename='./reportDMClashes.log', filemode='w')

dom3DM_ClashReporterLogger = logging.getLogger("reportDMClashes")
dom3DM_ClashReporterLogger.info("Initiating  reportDMClashes") 


class defs:
	item_open_entry_pat='^#new'
	item_type_line=item_open_entry_pat+'(\w+)\s*(\d+)'
	item_close_entry_pat='^#end'
	item_attribute_prefix='^#'
	item_attribute_value_line=item_attribute_prefix+'(\w+)\s+("?\.?\/?\w+.*)"?$'
	item_attribute_line=item_attribute_prefix+'(\w+)'
	name_line=item_attribute_prefix+'name'+'\s*"(.*)"'

class item:
	"""
		id - identifier
		name - name
		type - type (armor/weapon/unit ... )
		file - mode file
		line - line #
		attrs - additional attributs {attr_name:value,...}
		Ex:
	#newweapon 864
	#name "Warrior Shuriken"
	#dmg 10
	#att 2
	#def 0
	#len 0
	#range 12
	#ammo 2
	#nratt 2
	#nostr
	#sound 19
	#flyspr 111 1
	#rcost 3
	#armorpiercing
	#end

	"""

	m_id = 0
	m_name = ""
	m_type = ""
	m_file = ""
	m_line = 0
	m_attrs = {}

	def __init__(self, id=0 , name="item", type="none", file = "file.dm", line = 0, attrs={}):
		self.m_id = id;
		self.m_name = name;
		self.m_type = type;
		self.m_file = file;
		self.m_line = line;
		self.m_attrs = attrs;

	def __str__(self):
		return """
Mod entry Info:
ID: %d
Name: %s
Type: %s
File: %s
Line: %d
Attrs: %s
"""%(self.m_id, self.m_name, self.m_type, self.m_file, self.m_line, self.m_attrs)

class DomIII_DM_Clash_Reporter:
	"""
	Main worker class
	"""

	m_mod_items_list = {} # a dictionary of {(item ID, item type) : [item1, item2_id] }

	def parseMod(self, mod_file_name):
		"""
		Parse the mod & return items dictionary
		"""
		dom3DM_ClashReporterLogger.info("Parsing mod file %s"%mod_file_name) 
		mod_file = open(mod_file_name).readlines()


		"""
		states: init- start loading mod file, still haven't seen a single entry
			entry -  loading mod entry
			fin- finished processing the mod file
		"""
		states=['init', 'entry', 'fin'] 
		init,entry,fin = range(3)
		state=states[init]
		entries={} # mod entry/item dictionary
		temp_item=()
		line_counter=0

		for line in mod_file:
			line_counter+=1
			dom3DM_ClashReporterLogger.debug("Processing line %d:%s"%(line_counter,line))
############################ INIT State ######################
			if state == states[init]: #in init state 
				if re.match(defs.item_open_entry_pat, line ):
					state=states[entry] # found [ so parsing a mod entry
					dom3DM_ClashReporterLogger.debug("Found a mod entry. Switching to state %s "%(state))
					#cleanup item object
					temp_item = item()
					match = re.match(defs.item_type_line, line) #Note that #newspell entries don't have IDs and will be set to "fake" ID 0
					if match: #When dealing with a mod entry that has ID
						temp_item.m_type = match.group(1)
						temp_item.m_id = int(match.group(2))
					else: #When dealing with a mod entry that don't have ID - like spell
						temp_item.m_type = "spell"
						temp_item.m_id = 0

					temp_item.m_file = mod_file_name
					temp_item.m_line = line_counter
					dom3DM_ClashReporterLogger.info("Entry. Type: %s, ID: %d, %s:%d "%(temp_item.m_type, temp_item.m_id, temp_item.m_file, temp_item.m_line  ))
############################ INIT State ######################
############################ ENTRY State #####################
			if state == states[entry]: #in entry state 
				
				#match name & attrs
				if re.match(defs.name_line, line):
					match=re.match(defs.name_line, line)
					temp_item.m_name=match.group(1)

					dom3DM_ClashReporterLogger.debug("Found item name %s "%(temp_item.m_name))

				#attr value line
				elif re.match(defs.item_attribute_value_line, line):
					if re.match(defs.item_open_entry_pat, line ):
						dom3DM_ClashReporterLogger.debug("Skipping entry opening line since it was already parsed")
					else: #Real attr 
						match = re.match(defs.item_attribute_value_line, line)
						temp_item.m_attrs[match.group(1)] = match.group(2).strip()

						dom3DM_ClashReporterLogger.debug("Found item attr:value pair %s:%s "%(match.group(1), temp_item.m_attrs[match.group(1)] ))

				#End entry
				elif re.match(defs.item_close_entry_pat, line ):
					dom3DM_ClashReporterLogger.debug("Finished Gathering item data: %s."%temp_item)
	#match attrs and put in dictonary m_mod_items_list , if key exists add to list
					#add item to items dictionary. 
					if self.m_mod_items_list.has_key((temp_item.m_id, temp_item.m_type)):
						dom3DM_ClashReporterLogger.debug("Item %s ID + Type already exists in items dictionary, adding to list."%temp_item.m_name)
						self.m_mod_items_list[(temp_item.m_id, temp_item.m_type)].append(temp_item)
					else: #New item type with this ID
						dom3DM_ClashReporterLogger.debug("Item %s added to items dictionary"%temp_item.m_name)
						self.m_mod_items_list[(temp_item.m_id, temp_item.m_type)]= [temp_item]



					state=states[init] 
					dom3DM_ClashReporterLogger.debug("Found modentry end. Switch to state %s "%(state))
				#attr line
				elif re.match(defs.item_attribute_line, line):
					match = re.match(defs.item_attribute_line, line)
					temp_item.m_attrs[match.group(1)] = ""

					dom3DM_ClashReporterLogger.debug("Found item attr %s "%(match.group(1)) )
#				else:
#
#					dom3DM_ClashReporterLogger.debug("Found blank or comment line  %s"%(line)
############################ ENTRY State #####################


############################ FINIT State #####################
		state=states[fin]
		dom3DM_ClashReporterLogger.info("Finished processing mod file %s. State %s"%(mod_file_name, state))
############################ FINIT State #####################
				
				#raw print item dictionary

		#dom3DM_ClashReporterLogger.debug("Printing full items dictionary: %s"%(pickle.dumps(self.m_mod_items_list)))
		dom3DM_ClashReporterLogger.debug("Printing full items dictionary:")
#		for itr_item_key in self.m_mod_items_list.keys() :
#			for itr_item in self.m_mod_items_list[itr_item_key] :
#				dom3DM_ClashReporterLogger.debug("%s\n"%itr_item)
		for itr_item_key, itr_item_val  in self.m_mod_items_list.items():
			for itr_item in itr_item_val:
				dom3DM_ClashReporterLogger.debug("%s\n"%itr_item)
			
		#dom3DM_ClashReporterLogger.debug("Game list: %s"%(self.m_games))
############################ END OF mod lines parse loop  ####
		dom3DM_ClashReporterLogger.info("Complete mod file parsing")




	def parseMods(self):
		dom3DM_ClashReporterLogger.info("parsing the mod files into working memory. Files %s"%sys.argv)

		for file in sys.argv[1:]:
			self.parseMod(file)

		dom3DM_ClashReporterLogger.info("Complete mod files parsing")


	def reportConflicts(self):
		dom3DM_ClashReporterLogger.info("Reporting ID conflicts in the sepecified mod files")
		self.parseMods()
		#filter all non 0 ID items (i.e non spells) that appear in more than one mod file
		conflict_list = filter (lambda x: (len(self.m_mod_items_list[x]) > 1) and (self.m_mod_items_list[x][0].m_id !=0) , self.m_mod_items_list.keys() )
		for conf in conflict_list:
			#raw print
			#print "Warning found conflict %s"%self.m_mod_items_list[conf]

			#beutify print, item ID, + list of File:line, type name 
			dom3DM_ClashReporterLogger.info("Warning: found conflict! ID %d appears %d times."%(self.m_mod_items_list[conf][0].m_id, len(self.m_mod_items_list[conf])))
			print "Warning: found conflict! ID %d appears %d times."%(self.m_mod_items_list[conf][0].m_id, len(self.m_mod_items_list[conf]))
			for itr in self.m_mod_items_list[conf] :
				dom3DM_ClashReporterLogger.info("At %s:%d, as type: %s, name %s"%(itr.m_file, itr.m_line, itr.m_type, itr.m_name) )
				print "At %s:%d, as type: %s, name %s"%(itr.m_file, itr.m_line, itr.m_type, itr.m_name) 

		dom3DM_ClashReporterLogger.info("Completed reporting conflicts")



if __name__ == '__main__':
	DMCR = DomIII_DM_Clash_Reporter();
	DMCR.reportConflicts()

