/* LICENSED UNDER THE TERMS OF THE LGPL
 * http://www.gnu.org/copyleft/lesser.html
 */

/*
 * MainForm.java
 *
 * Created on February 18, 2005, 6:29 PM
 */

package templatizer;
import java.awt.FileDialog;
import java.io.*;
import javax.swing.JOptionPane;
import java.util.ArrayList;

/**
 *
 * @author  koliset
 */
public class MainForm extends javax.swing.JFrame {
    
    private String superGlobals = "";
    private ArrayList<Function> functions = new ArrayList<Function>();
    private String outFileName = null;
    private DataOutputStream outFile = null;
    private boolean allDone = false;
    private String calcPath = "calc";
    private String[] args = new String[0];
    
    
    /** Creates new form MainForm */
    public MainForm() {
	initComponents();
    }
    
    /** Creates new form MainForm */
    public MainForm(String[] args) {
	initComponents();
	this.args = args;
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        pbRecordProgress = new javax.swing.JProgressBar();
        pbProgress = new javax.swing.JProgressBar();
        pnlStatus = new javax.swing.JPanel();
        lblRecordStatus = new javax.swing.JLabel();
        lblStatus = new javax.swing.JLabel();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("SE4 Templatizer v3.0.3");
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowOpened(java.awt.event.WindowEvent evt) {
                formWindowOpened(evt);
            }
        });

        pbRecordProgress.setMinimumSize(new java.awt.Dimension(320, 16));
        pbRecordProgress.setPreferredSize(new java.awt.Dimension(480, 16));
        getContentPane().add(pbRecordProgress, java.awt.BorderLayout.NORTH);

        pbProgress.setMinimumSize(new java.awt.Dimension(320, 16));
        pbProgress.setPreferredSize(new java.awt.Dimension(480, 16));
        getContentPane().add(pbProgress, java.awt.BorderLayout.CENTER);

        pnlStatus.setLayout(new java.awt.BorderLayout());

        pnlStatus.setMinimumSize(new java.awt.Dimension(320, 16));
        pnlStatus.setPreferredSize(new java.awt.Dimension(480, 16));
        lblRecordStatus.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        lblRecordStatus.setText("Record Status");
        lblRecordStatus.setAlignmentX(0.5F);
        lblRecordStatus.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        lblRecordStatus.setMinimumSize(new java.awt.Dimension(240, 16));
        lblRecordStatus.setPreferredSize(new java.awt.Dimension(240, 16));
        pnlStatus.add(lblRecordStatus, java.awt.BorderLayout.WEST);

        lblStatus.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        lblStatus.setText("Templatizer Status");
        lblStatus.setAlignmentX(0.5F);
        lblStatus.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        lblStatus.setMinimumSize(new java.awt.Dimension(240, 16));
        lblStatus.setPreferredSize(new java.awt.Dimension(240, 16));
        pnlStatus.add(lblStatus, java.awt.BorderLayout.EAST);

        getContentPane().add(pnlStatus, java.awt.BorderLayout.SOUTH);

        pack();
    }//GEN-END:initComponents
    
	private void formWindowOpened(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowOpened
	    
	    //read in the path to calc
	    try {
		DataInputStream inFile = new DataInputStream(new FileInputStream("calc-path.txt"));
		calcPath = inFile.readLine();
		inFile.close();
	    } catch (FileNotFoundException ex) {
		JOptionPane.showMessageDialog(this, "Could not load calc-path.txt. Using default calc path of \"calc\"." + "\n" + ex.getStackTrace()[0]);
	    } catch (IOException ex) {
		JOptionPane.showMessageDialog(this, "Could not load calc-path.txt. Using default calc path of \"calc\"." + "\n" + ex.getStackTrace()[0]);
	    }
	    
	    lblStatus.setText("Awaiting user input...");
	    
	    // Get input filename
	    String inFileName;
	    if (args.length > 0)
		inFileName = args[0];
	    else {
		FileDialog dlg1 = new FileDialog(this, "Please select a template file", FileDialog.LOAD);
		dlg1.setVisible(true);
		inFileName = dlg1.getDirectory() + dlg1.getFile();
		
		// exit if nothing is specified
		if (dlg1.getFile() == null)
		    Runtime.getRuntime().exit(1);
	    }
	    
	    lblStatus.setText("Reading input file...");
	    
	    // Read the template file
	    ArrayList<String> input = new ArrayList<String>();
	    try {
		DataInputStream inFile = new DataInputStream(new FileInputStream(inFileName));
		boolean done = false;
		do {
		    String line = inFile.readLine();
		    if (line == null)
			done = true;
		    else
			input.add(line);
		} while (!done);
		inFile.close();
	    } catch (FileNotFoundException ex) {
		JOptionPane.showMessageDialog(this, ex.getMessage() + "\n" + ex.getStackTrace()[0]);
		Runtime.getRuntime().exit(1);
	    } catch (IOException ex) {
		JOptionPane.showMessageDialog(this, ex.getMessage() + "\n" + ex.getStackTrace()[0]);
		Runtime.getRuntime().exit(1);
	    }
	    
	    lblStatus.setText("Awaiting user input...");
	    
	    // Get output filename
	    FileDialog dlg2 = new FileDialog(this, "Please select an output file", FileDialog.SAVE);
	    dlg2.setDirectory(new File(inFileName).getParent()); // same path as input, by default
	    dlg2.setVisible(true);
	    outFileName = dlg2.getDirectory() + dlg2.getFile();
	    
	    // exit if nothing specified
	    if (dlg2.getFile() == null)
		Runtime.getRuntime().exit(1);
	    
	    // Check for overwrite
	    if (new File(outFileName).exists() && !(JOptionPane.showConfirmDialog(this, "Really overwrite " + outFileName + "?") == JOptionPane.YES_OPTION))
		Runtime.getRuntime().exit(1);
	    
	    // delete file before writing to it
	    new File(outFileName).delete();
	    
	    lblStatus.setText("Converting input file to records...");
	    
	    // create records from lines
	    ArrayList<Record> records = new ArrayList<Record>();
	    Record curRecord = new Record();
	    
	    for (String line : input) {
		if (line.equals("") || line == null) {
		    // end of record; save and begin new record
		    records.add(curRecord);
		    curRecord = new Record();
		} else if (line.contains(":=") && line.length() >= 2 && line.trim().substring(0, 1) != "//") {
		    int separatorPos = line.indexOf(":=");
		    String key = line.substring(0, separatorPos - 1).trim();
		    String value = line.substring(separatorPos + 2, line.length()).trim();
		    try {
			curRecord.add(new Field(line));
		    } catch (Exception ex) {
			JOptionPane.showMessageDialog(this, ex.getMessage() + "\n" + ex.getStackTrace()[0]);
		    }
		}
	    }
	    if (curRecord.size() > 0)
		records.add(curRecord);
	    
	    // do the templatizing
	    pbProgress.setMaximum(records.size());
	    int recNum = 0;
	    
	    try {
		// open the file
		outFile = new DataOutputStream(new FileOutputStream(outFileName));
		
		// write begin tag
		outFile.writeBytes("*BEGIN*\r\n\r\n\r\n");
		
		// write records
		for (Record record : records) {
		    // update status
		    lblStatus.setText("Processing template " + (recNum + 1) + " of " + records.size());
		    pbProgress.setValue(recNum);
		    // this templatizing is very processor intensive, so make sure
		    // the status bar & label repainting gets done!
		    paintEverything();
		    
		    boolean hasFields = false;
		    // do initial parsing for functions and grids
		    for (int i = 0; i < record.size(); i++) {
			Field field = record.get(i);
			String key = field.getKey();
			String value = removeComment(field.getValue());
			if (key.length() >= 1 && key.charAt(0) == '*') {
			    // define a global function
			    // e.g. "*add(x,y) := x + y"
			    functions.add(new Function(Function.Type.GLOBAL, new Field(field.getKey().substring(1), removeComment(field.getValue()))));
			    record.remove(field);
			    i--;
			} else if (key.length() >= 1 && key.charAt(0) == '@') {
			    // define a local function
			    // e.g. "@add(x,y) := x + y"
			    functions.add(new Function(Function.Type.LOCAL, new Field(field.getKey().substring(1), removeComment(field.getValue()))));
			    record.remove(field);
			    i--;
			} else if (key.length() >= 1 && key.charAt(0) == '#') {
			    // define a tech grid area
			    // e.g. "#damage_enhance := 1..5"
			    if (!value.contains("..")) {
				JOptionPane.showMessageDialog(this, "Tech grid area value " + value + " does not contain two consecutive periods.");
				Runtime.getRuntime().exit(1);
			    }
			    int periodPos = value.indexOf("..");
			    Integer min = Integer.parseInt(evaluate(value.substring(0, periodPos), false, 0));
			    Integer max = Integer.parseInt(evaluate(removeComment(value.substring(periodPos + 2, value.length())), false, 0));
			    functions.add(new Function(Function.Type.GRID, new TechGridArea(key.substring(1), min.toString(), max.toString(), min)));
			    record.remove(field);
			    i--;
			} else {
			    // this is a record with actual fields, not just functions!
			    hasFields = true;
			}
		    }
		    
		    // do the gridding
		    if (hasFields)
			gridRecord(record);
		    
		    // increment the record counter so the progress bar moves
		    recNum++;
		    
		    // clean up the local stuff
		    for (int i = 0; i < functions.size(); i++) {
			Function function = functions.get(i);
			if (function.getType() != Function.Type.GLOBAL) {
			    functions.remove(function);
			    i--;
			}
		    }
		}
		
		// write end tag
		outFile.writeBytes("*END*");
		
		// close the output file
		outFile.close();
		
	    } catch (IOException ex) {
		JOptionPane.showMessageDialog(this, ex.getMessage() + "\n");
		try {
		    outFile.close();
		} catch (IOException ex2) {
		    JOptionPane.showMessageDialog(this, ex2.getMessage() + "\n");
		    Runtime.getRuntime().exit(1);
		}
		
		// quit
		Runtime.getRuntime().exit(1);
	    }
	    Runtime.getRuntime().exit(0);
	}//GEN-LAST:event_formWindowOpened
	
	/**
	 * Gets all the functions from a function list in a suitable format for calc
	 */
	private String getAllFunctions() {
	    String result = "";
	    for (Function function : functions) {
		// is it a regular function or a tech grid area?
		if (function.getType() == Function.Type.GRID) {
		    TechGridArea grid = (TechGridArea)function.getField();
		    result += grid.getName() + " = " + grid.getCurrent() + ";";
		} else {
		    Field field = (Field)function.getField();
		    // is it a function or a variable?
		    if (field.getKey().contains("("))
			result += "define " + field.getKey() + " = " + field.getValue() + ";";
		    else
			result += field.getKey() + " = " + field.getValue() + ";";
		}
	    }
	    return result;
	}
	
	/**
	 * Evaluates any expressions contained in square brackets in a string.
	 */
	private String evaluate(String value, boolean isWeaponDamage, int range) {
	    // check for expressions enclosed in square brackets
	    boolean inExpression = false;
	    String result = "";
	    String text = "";
	    int bracketsLevel = 0; // keeps track of nested square brackets
	    boolean lastWasBackslash = false; // keeps track of escape sequences
	    for (char c : value.toCharArray()) {
		if (c == (char)'[') {
		    if (bracketsLevel == 0 && !lastWasBackslash) {
			// end old text
			result += text;
			text = "";
			inExpression = true;
		    } else
			text += c;
		    bracketsLevel++;
		    lastWasBackslash = false;
		} else if (c ==  (char)']' && inExpression) {
		    if (bracketsLevel == 1) {
			// calculate the expression's value
			int loopMin = 0;
			int loopMax = 0;
			final String args;
			if (isWeaponDamage)
			    args = getAllFunctions() + "range = " + range + ";" + text;
			else
			    args = getAllFunctions() + text;
			String littleResult = "";
			try {
			    ProcessBuilder pb = new ProcessBuilder(calcPath,"-c", "-d", "read", "templatizer-functions.txt", "\n", args);
			    // so stderr can be read thru stdout
			    pb.redirectErrorStream(true);
			    Process p = pb.start();
			    DataInputStream resultStream = new DataInputStream(p.getInputStream());
			    littleResult = resultStream.readLine().trim();
			    // just in case there was an error
			    // don't want to clutter up memory with a bunch of processes
			    p.destroy();
			    
			    // remove quotes
			    if (littleResult != null && littleResult.length() > 0 && littleResult.charAt(0) == '"')
				littleResult = littleResult.substring(1, littleResult.length() - 1);
			} catch (Exception ex) {
			    JOptionPane.showMessageDialog(this, ex.getMessage());
			    Runtime.getRuntime().exit(1);
			}
			
			// no negative weapon damages!
			if (isWeaponDamage && result.charAt(0) == '-')
			    result += "0";
			else
			    result += littleResult;
			text = "";
			inExpression = false;
		    } else
			text += c;
		    bracketsLevel--;
		    lastWasBackslash = false;
		    
		} else if (c == '\\' && !lastWasBackslash)
		    lastWasBackslash = true;
		else {
		    text += c;
		    lastWasBackslash = false;
		}
	    }
	    result += text;
	    return result;
	}
	
	/*
	 * Removes any trailing "//" comments from a string which may contain
	 * expressions; double slashes inside expressions are left alone.
	 */
	private String removeComment(String text) {
	    String result = "";
	    int bracketsLevel = 0;
	    boolean lastWasSlash = false;
	    for (char c : text.toCharArray()) {
		if (c == (char)'[') {
		    bracketsLevel++;
		    lastWasSlash = false;
		} else if (c == (char)']') {
		    bracketsLevel--;
		    lastWasSlash = false;
		}
		if (c == (char)'/' && bracketsLevel <= 0) {
		    // double slashes outside an expression mean a comment
		    if (lastWasSlash) {
			result = result.substring(0, text.length() - 2);
			break;
		    } else
			lastWasSlash = true;
		    break;
		} else {
		    result += c;
		    lastWasSlash = false;
		}
	    }
	    return result;
	}
	
	/*
	 * Grids a tech record
	 */
	private void gridRecord(Record record) throws IOException {
	    // find out how many iterations there will be
	    int iterations = 1;
	    ArrayList<TechGridArea> techGrids = getGrids();
	    for (TechGridArea grid : techGrids) {
		int max = Integer.parseInt(evaluate(grid.getMaximum(), false, 0));
		int min = Integer.parseInt(evaluate(grid.getMinimum(), false, 0));
		int levels = max - min + 1;
		iterations *= levels;
	    }
	    
	    // set up the progress bar
	    pbRecordProgress.setMaximum(iterations);
	    
	    // go until there are no more iterations to do
	    for (int i = 0; i < iterations; i++) {
		// set the progress bar and label
		pbRecordProgress.setValue(i);
		lblRecordStatus.setText("Generating record " + (i+1) + " of " + iterations + " for this template");
		// this templatizing is very processor intensive, so make sure
		// the status bar & label repainting gets done!
		paintEverything();
		
		// do the gridding
		NumberRange weaponRange = new NumberRange("1", "20");
		for (Field field : record) {
		    String key = field.getKey();
		    String value = removeComment(field.getValue());
		    if (key.length() >= 1 && key.charAt(0) == '~') {
			// define the weapon's minimum and maximum effective range
			// e.g. "~ := 1..8"
			if (!value.contains("..")) {
			    JOptionPane.showMessageDialog(this, "Weapon range value " + value + " does not contain two consecutive periods.");
			    Runtime.getRuntime().exit(1);
			}
			int periodPos = value.indexOf("..");
			weaponRange = new NumberRange(value.substring(0, periodPos), value.substring(periodPos + 2, value.length() - 1));
		    } else if (key.equals("Weapon Damage At Rng")) {
			// special case: define extra variables
			outFile.writeBytes(key + " := ");
			int minRange = Integer.parseInt(evaluate(weaponRange.getMinimum(), false, 0));
			int maxRange = Integer.parseInt(evaluate(weaponRange.getMaximum(), false, 0));
			for (int range = 1; range < minRange; range++)
			    outFile.writeBytes("0 ");
			for (int range = minRange; range <= maxRange; range++)
			    outFile.writeBytes(evaluate(value, true, range) + " ");
			for (int range = maxRange + 1; range < (minRange > 20 ? 21 : 20); range++)
			    outFile.writeBytes("0 ");
			outFile.writeBytes("\r\n");
		    } else {
			// evaluate any expressions and write the output line
			outFile.writeBytes(key + " := " + evaluate(value, false, 0)  + "\r\n");
		    }
		}
		
		// write blank line between records
		outFile.writeBytes("\r\n");
		
		// increment the tech grid areas as necessary
		try {
		    incrementGrid(techGrids.size() - 1);
		} catch (IllegalArgumentException ex) {
		    // do nothing, this just means we've hit the end
		    // of the template
		}
	    }
	}
	
	/*
	 * Increments a tech grid area
	 */
	private void incrementGrid(int num) throws IllegalArgumentException {
	    if (num < 0)
		throw new IllegalArgumentException("Tried to increment the first tech grid area past its maximum.");
	    ArrayList<TechGridArea> grids = getGrids();
	    TechGridArea grid = grids.get(num);
	    if (grid.getCurrent() >= Integer.parseInt(evaluate(grid.getMaximum(), false, 0))) {
		for (int i = num; i < grids.size(); i++) {
		    TechGridArea otherGrid = grids.get(i);;
		    otherGrid.setCurrent(Integer.parseInt(evaluate(otherGrid.getMinimum(), false, 0)));
		}
		incrementGrid(num - 1);
	    } else
		grid.setCurrent(grid.getCurrent() + 1);
	}
	
	/*
	 * Repaints everything
	 * Templatizing is a very processor intensive process
	 * So the templatizer might "forget" to repaint the status indicators
	 */
	private void paintEverything() {
	    this.paint(this.getGraphics());
	    pnlStatus.paint(pnlStatus.getGraphics());
	    lblStatus.paint(lblStatus.getGraphics());
	    lblRecordStatus.paint(lblRecordStatus.getGraphics());
	}
	
	/*
	 * Gets the tech grids from the function list
	 */
	private ArrayList<TechGridArea> getGrids() {
	    ArrayList<TechGridArea> techGrids = new ArrayList<TechGridArea>();
	    for (Function function : functions) {
		if (function.getType() == Function.Type.GRID)
		    techGrids.add((TechGridArea)function.getField());
	    }
	    return techGrids;
	}
	
	/**
	 * @param args the command line arguments
	 */
	public static void main(final String[] args) {
	    java.awt.EventQueue.invokeLater(new Runnable() {
		public void run() {
		    new MainForm(args).setVisible(true);
		}
	    });
	}
	
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel lblRecordStatus;
    private javax.swing.JLabel lblStatus;
    private javax.swing.JProgressBar pbProgress;
    private javax.swing.JProgressBar pbRecordProgress;
    private javax.swing.JPanel pnlStatus;
    // End of variables declaration//GEN-END:variables
    
}
