//============================================================================
//============================================================================

function findBounds(canvas) 
{
  const ctx = canvas.getContext('2d');
  const width = canvas.width;
  const height = canvas.height;
  const imageData = ctx.getImageData(0, 0, width, height);
  const data = imageData.data;

  let boundRight = -1;
  let boundBottom = -1;

  // Iterate through each pixel in the canvas
  for (let y = 0; y < height; y++) {
    for (let x = 0; x < width; x++) {
      const index = (y * width + x) * 4;
      const r = data[index];
      const g = data[index + 1];
      const b = data[index + 2];

      // Check if the pixel is not (255, 225, 225)
      if (r !== 255 || g !== 225 || b !== 225) {
        // Update the rightmost bound
        if (x > boundRight) {
          boundRight = x;
        }
        // Update the bottommost bound
        if (y > boundBottom) {
          boundBottom = y;
        }
      }
    }
  }

//fix for pixel indexing to bring it in line, etc.
//boundRight = boundRight +1;
//boundBottom = boundBottom +1;
//turns out we don't actually need to do this lol.

  return { boundRight, boundBottom };
}

//============================================================================
//============================================================================

    function readImageHeader(dataView, offset, imageNumber) 
	{
      // SHP Image Header: 24 bytes long
      const height = dataView.getUint16(offset + 0, true) +1; // Height is stored as lines - 1
      const width = dataView.getUint16(offset + 2, true) +1;
      const var1 = dataView.getInt32(offset + 4, true); // Use getInt32 to allow for negative numbers
      const xStart = dataView.getInt32(offset + 8, true); // Use getInt32 to allow for negative numbers
      const yStart = dataView.getInt32(offset + 12, true); // Use getInt32 to allow for negative numbers
      const xEnd = dataView.getInt32(offset + 16, true); // Use getInt32 to allow for negative numbers
      const yEnd = dataView.getInt32(offset + 20, true); // Use getInt32 to allow for negative numbers
	  
	// Include the image number in the log message
	conditionalLog(`Image ${imageNumber} Header - Height: ${height}, Width: ${width}, Var1: ${var1}, xStart: ${xStart}, yStart: ${yStart}, xEnd: ${xEnd}, yEnd: ${yEnd}`); // only if verbose logging included.
      
      return { height, width, var1, xStart, yStart, xEnd, yEnd };
    }
	
//============================================================================
//============================================================================

// Define the off-screen canvas outside of the event listeners so it remains in scope
let offScreenCanvas, offScreenCtx;

//initalize bounding box variable here so we can read it across multiple functions.
let boundingbox;

//initalize the icon index so that we can pass it to other windows...
let selectedIconIndex;

//============================================================================
//============================================================================

document.getElementById("writeSHPheaders").addEventListener("click", function() {
  // Log all child canvases in the canvasContainer
  const canvasContainer = document.getElementById('canvasContainer');
  const childCanvases = canvasContainer.querySelectorAll('canvas');

  if (!originalSHPFileArrayBuffer) {
    console.error("Original SHP file buffer is not loaded.");
    return;
  }

  if (childCanvases.length === 0) {
    console.error("No canvases found. Make sure canvases are loaded properly.");
    return;
  }

  // Create a new ArrayBuffer to modify, based on the original, to avoid access issues
  let modifiedArrayBuffer = originalSHPFileArrayBuffer.slice(0);
  const dataView = new DataView(modifiedArrayBuffer);

  // Get the number of images from offset 4
  const numImages = dataView.getUint32(4, true); // Assuming little-endian byte order
  //console.log(`Number of images in SHP file: ${numImages}`);

  let tableOffset = 8;

  // Loop through all canvases (assuming each canvas corresponds to an image in the SHP)
  for (let i = 0; i < childCanvases.length; i++) {
	  
    // Subtract 1 from the index to align with correct zero-based indexing
    const adjustedIndex = i - 1;

    if (adjustedIndex < 0) {
      console.error(`Invalid index after adjustment: ${adjustedIndex}`);
      continue;
    }

    const canvas = childCanvases[adjustedIndex];
    
    // Ensure the dataset values are properly set before proceeding
    const width = Number(canvas.dataset.width); // Use Number to convert, which handles negative numbers
    const height = Number(canvas.dataset.height);
    const var1 = Number(canvas.dataset.var1);
    const xStart = Number(canvas.dataset.xStart);
    const yStart = Number(canvas.dataset.yStart);
    const xEnd = Number(canvas.dataset.xEnd);
    const yEnd = Number(canvas.dataset.yEnd);

    //console.log(`Canvas ${adjustedIndex + 1} - width: ${width}, height: ${height}, var1: ${var1}, xStart: ${xStart}, yStart: ${yStart}, xEnd: ${xEnd}, yEnd: ${yEnd}`);

    // Get the SHP offset for this image from the table (at tableOffset)
    const shpOffset = dataView.getUint32(tableOffset, true);
    if (shpOffset < 0 || shpOffset >= modifiedArrayBuffer.byteLength) {
      console.error(`Invalid SHP offset found: ${shpOffset} for canvas index ${adjustedIndex}`);
      return;
    }
    //console.log(`Modifying SHP Image #${adjustedIndex + 1} at offset ${shpOffset}`);

    // Modify the header for the current SHP image
    try {
      dataView.setInt16(shpOffset, height, true); // Height at offset 0 (signed)
      dataView.setInt16(shpOffset + 2, width, true); // Width at offset 2 (signed)
      dataView.setInt32(shpOffset + 4, var1, true); // Var1 at offset 4 (signed)
      dataView.setInt32(shpOffset + 8, xStart, true); // xStart at offset 8 (signed)
      dataView.setInt32(shpOffset + 12, yStart, true); // yStart at offset 12 (signed)
      dataView.setInt32(shpOffset + 16, xEnd, true); // xEnd at offset 16 (signed)
      dataView.setInt32(shpOffset + 20, yEnd, true); // yEnd at offset 20 (signed)

      // Read back to verify the modification
      const verifyHeight = dataView.getInt16(shpOffset, true);
      const verifyWidth = dataView.getInt16(shpOffset + 2, true);
      //console.log(`Verified height at offset ${shpOffset}: ${verifyHeight}, width: ${verifyWidth}`);

    } catch (error) {
      console.error(`Error modifying SHP Image #${adjustedIndex + 1} at offset ${shpOffset}:`, error);
      return;
    }

    // Move to the next table entry (each entry is 8 bytes)
    tableOffset += 8;
  }

  console.log("All child canvases data has been written to the modified SHP file buffer.");

  // Save the modified SHP file
  const blob = new Blob([modifiedArrayBuffer], { type: "application/octet-stream" });
  const link = document.createElement("a");
  link.href = URL.createObjectURL(blob);
  link.download = "modified_SHP_file.shp";
  link.click();

  console.log("Modified SHP file saved as modified_SHP_file.shp");
});


//============================================================================
//============================================================================

// Write current header data to the selected child canvas....
document.getElementById('writeHeaderSingle').addEventListener('click', function() 
{

	// Log all child canvases in the canvasContainer
	const canvasContainer = document.getElementById('canvasContainer');
	const childCanvases = canvasContainer.querySelectorAll('canvas');

	//console.log("Child canvases found:", childCanvases);
	//childCanvases.forEach((canvas, index) => {
	//  console.log(`Canvas ${index}:`, canvas);
	//});

	// Select the child canvas based on the index from selectedIconIndex
	  const childCanvas = childCanvases[selectedIconIndex - 1]; // SUBTRACT ONE TO ALIGN WITH CORRECT INDEX.

	  if (!childCanvas) {
		alert('Child canvas not found!');
		return;
	  }
    
  // Grab new header data from the window...
  const newHeader = 
  {
    var1: document.getElementById('mainVar1').value,
    xStart: document.getElementById('mainXstart').value,
    yStart: document.getElementById('mainYstart').value,
    xEnd: document.getElementById('mainXend').value,
    yEnd: document.getElementById('mainYend').value
  };

  // Log the new values to ensure they are correct
  //console.log("var1:" + newHeader.var1);
  //console.log("xStart:" + newHeader.xStart);
  //console.log("yStart:" + newHeader.yStart);
  //console.log("xEnd:" + newHeader.xEnd);
  //console.log("yEnd:" + newHeader.yEnd);
    
  // Update the dataset for the specific child canvas
  childCanvas.dataset.var1 = newHeader.var1; 
  childCanvas.dataset.xStart = newHeader.xStart;
  childCanvas.dataset.yStart = newHeader.yStart;
  childCanvas.dataset.xEnd = newHeader.xEnd;
  childCanvas.dataset.yEnd = newHeader.yEnd;

  //alert('Dataset updated for the child canvas at index ' + (selectedIconIndex-1) + '!'); // SUBTRACT ONE TO ALIGN WITH CORRECT INDEX.
});

//=============================================================================

// Write current header data to multiple child canvases starting from selectedIconIndex....
document.getElementById('writeHeaderMultiple').addEventListener('click', function() 
{
  // Adjust index by subtracting 1
  const adjustedIndex = selectedIconIndex - 1;

  // Ask the user how many icons to update
  const numberOfIcons = parseInt(prompt("How many icons do you wish to write this to?", "1"), 10);

  if (isNaN(numberOfIcons) || numberOfIcons <= 0) {
    alert('Invalid number of icons!');
    return;
  }

  // Log all child canvases in the canvasContainer
  const canvasContainer = document.getElementById('canvasContainer');
  const childCanvases = canvasContainer.querySelectorAll('canvas');

  //console.log("Child canvases found:", childCanvases);
  //childCanvases.forEach((canvas, index) => {
 //   console.log(`Canvas ${index}:`, canvas);
 // });

  // Check if the adjusted index is valid and if there are enough canvases to update
  if (adjustedIndex < 0 || adjustedIndex >= childCanvases.length) {
    alert('Invalid canvas index selected!');
    return;
  }

  const endIndex = adjustedIndex + numberOfIcons - 1;

  if (endIndex >= childCanvases.length) {
    alert('Not enough canvases available! You can update up to ' + (childCanvases.length - adjustedIndex) + ' canvases.');
    return;
  }

  // Grab new header data from the window...
  const newHeader = 
  {
    var1: document.getElementById('mainVar1').value,
    xStart: document.getElementById('mainXstart').value,
    yStart: document.getElementById('mainYstart').value,
    xEnd: document.getElementById('mainXend').value,
    yEnd: document.getElementById('mainYend').value
  };

  // Log the new values to ensure they are correct
  //console.log("var1:" + newHeader.var1);
  //console.log("xStart:" + newHeader.xStart);
  //console.log("yStart:" + newHeader.yStart);
  //console.log("xEnd:" + newHeader.xEnd);
  //console.log("yEnd:" + newHeader.yEnd);

  // Loop through and update the dataset for each selected child canvas
  for (let i = adjustedIndex; i <= endIndex; i++) {
    const childCanvas = childCanvases[i];
    if (childCanvas) {
      // Update the dataset for the specific child canvas
      childCanvas.dataset.var1 = newHeader.var1; 
      childCanvas.dataset.xStart = newHeader.xStart;
      childCanvas.dataset.yStart = newHeader.yStart;
      childCanvas.dataset.xEnd = newHeader.xEnd;
      childCanvas.dataset.yEnd = newHeader.yEnd;

      //console.log('Dataset updated for child canvas at index ' + i);
    }
  }

  //alert('Dataset updated for ' + numberOfIcons + ' canvases starting from index ' + adjustedIndex + '!');
});

//============================================================================

// Write current header data to every other child canvas starting from selectedIconIndex....
document.getElementById('writeHeaderMultipleAlternating').addEventListener('click', function() 
{
  // Ask the user how many icons to update
  const numberOfIcons = parseInt(prompt("How many icons do you wish to write this to (every other icon)?", "1"), 10);

  if (isNaN(numberOfIcons) || numberOfIcons <= 0) {
    alert('Invalid number of icons!');
    return;
  }

  // Log all child canvases in the canvasContainer
  const canvasContainer = document.getElementById('canvasContainer');
  const childCanvases = canvasContainer.querySelectorAll('canvas');

  // console.log("Child canvases found:", childCanvases);
  // childCanvases.forEach((canvas, index) => {
  //  console.log(`Canvas ${index}:`, canvas);
  //});

  // Subtract 1 from selectedIconIndex to account for 1-based index
  let startIndex = selectedIconIndex - 1;

  // Check if the startIndex is valid
  if (startIndex < 0 || startIndex >= childCanvases.length) {
    alert('Invalid canvas index selected!');
    return;
  }

  // Loop through and update the dataset for every other child canvas
  let count = 0;
  for (let i = startIndex; count < numberOfIcons && i < childCanvases.length; i += 2) {
    const childCanvas = childCanvases[i];
    if (childCanvas) {
      // Grab new header data from the window...
      const newHeader = 
      {
        var1: document.getElementById('mainVar1').value,
        xStart: document.getElementById('mainXstart').value,
        yStart: document.getElementById('mainYstart').value,
        xEnd: document.getElementById('mainXend').value,
        yEnd: document.getElementById('mainYend').value
      };

      // Log the new values to ensure they are correct
      //console.log("Updating canvas at index " + i);
      //console.log("var1:" + newHeader.var1);
      //console.log("xStart:" + newHeader.xStart);
      //console.log("yStart:" + newHeader.yStart);
      //console.log("xEnd:" + newHeader.xEnd);
      //console.log("yEnd:" + newHeader.yEnd);
        
      // Update the dataset for the specific child canvas
      childCanvas.dataset.var1 = newHeader.var1; 
      childCanvas.dataset.xStart = newHeader.xStart;
      childCanvas.dataset.yStart = newHeader.yStart;
      childCanvas.dataset.xEnd = newHeader.xEnd;
      childCanvas.dataset.yEnd = newHeader.yEnd;

      count++; // Increment count of updated icons
    }
  }

  //alert('Dataset updated for ' + numberOfIcons + ' canvases starting from index ' + selectedIconIndex + ' (every other canvas)!');
});

//============================================================================

// Handle "Show Header Right Click Info" action (triggered via right-click or other event)
document.getElementById('showHeaderInfo').addEventListener('click', function() {
  const contextMenu = document.getElementById('customContextMenu');
  const canvas = contextMenu.currentCanvas; // Get the clicked canvas

  if (canvas) {
    // Retrieve the header data stored in the canvas's dataset (populated during the image load)
    const headerInfo = {
      width: parseInt(canvas.dataset.width),
      height: parseInt(canvas.dataset.height),
      var1: parseInt(canvas.dataset.var1),
      xStart: parseInt(canvas.dataset.xStart),
      yStart: parseInt(canvas.dataset.yStart),
      xEnd: parseInt(canvas.dataset.xEnd),
      yEnd: parseInt(canvas.dataset.yEnd)
    };
	
	selectedIconIndex = parseInt(canvas.dataset.index); // get the index of the icon
	console.log("Icon Index:" + selectedIconIndex); //log to canvas

    // Populate the form fields in the right column with the header data
    document.querySelector('input[value="${headerInfo.height}"]').value = headerInfo.height;
    document.querySelector('input[value="${headerInfo.width}"]').value = headerInfo.width;
    document.querySelector('input[value="${headerInfo.var1}"]').value = headerInfo.var1;
    document.querySelector('input[value="${headerInfo.xStart}"]').value = headerInfo.xStart;
    document.querySelector('input[value="${headerInfo.yStart}"]').value = headerInfo.yStart;
    document.querySelector('input[value="${headerInfo.xEnd}"]').value = headerInfo.xEnd;
    document.querySelector('input[value="${headerInfo.yEnd}"]').value = headerInfo.yEnd;

    // Display the pop-up
    const popup = document.getElementById('headerInfoPopup');
    popup.style.display = 'block';

	//we now analyze the original image to find bounding box, from a known clean version.
	boundingbox = findBounds(canvas);
	//console.log('BoundRight:', boundingbox.boundRight, 'BoundBottom:', boundingbox.boundBottom);

    // Create an off-screen canvas to store the original image and scale it up by 4x for the window
    offScreenCanvas = document.createElement('canvas');
    offScreenCtx = offScreenCanvas.getContext('2d');
    offScreenCanvas.width = headerInfo.width * 4; // scale up by 4x
    offScreenCanvas.height = headerInfo.height * 4; //scale up by 4x
	offScreenCtx.imageSmoothingEnabled = false; // Disable image smoothing
	offScreenCtx.scale(4, 4);
	offScreenCtx.drawImage(canvas, 0, 0);
	//dumpCanvasAsPNG(offScreenCanvas, 'offScreenCanvas_debug.png'); // Example: Dumps the scaled canvas

    // We now draw the initial canvas on the HTML pop up window...
    const scaledCanvas = document.getElementById('scaledImageCanvas');
    const scaledCtx = scaledCanvas.getContext('2d');
    const scaledWidth = headerInfo.width * 4; // Set the scaled canvas size (4x original size)
    const scaledHeight = headerInfo.height * 4; // Set the scaled canvas size (4x original size)
    scaledCanvas.width = scaledWidth;
    scaledCanvas.height = scaledHeight;
	scaledCtx.imageSmoothingEnabled = false; // Disable image smoothing
	scaledCtx.clearRect(0, 0, scaledWidth, scaledHeight); // Clear before redrawing
    scaledCtx.drawImage(offScreenCanvas, 0, 0);
	//dumpCanvasAsPNG(scaledCanvas, 'scaledCanvas_debug.png'); // Example: Dumps the scaled canvas
	
    // Add the cross drawing functionality on the scaled canvas (scale coordinates by 4x)
    drawInteractiveSPHeaderCross(scaledCtx, 0, headerInfo.width * 4, headerInfo.height * 4, headerInfo.xStart, headerInfo.yStart, headerInfo.xEnd, headerInfo.yEnd);
  }

  // Hide the custom context menu after the action
  document.getElementById('customContextMenu').style.display = 'none';
});

//============================================================================
//============================================================================

// Function to draw a cyan blob (8x8 pixels at 4x scale) based on SHP header data
function drawInteractiveSPHeaderCross(ctx, index, width, height, xStart, yStart, xEnd, yEnd) {
  // Translate negative coordinates to positive by adding width and height (scaled by 4x)
  let translatedX = Math.abs(xStart) * 4; // Multiply by 4 for the scale
  let translatedY = Math.abs(yStart) * 4; // Multiply by 4 for the scale

  // Set the fill color to Cyan (0, 255, 255)
  ctx.fillStyle = 'rgb(0, 255, 255)';

  // Draw the horizontal line of the cross (12 pixels long at 4x scale, centered on translatedX)
  ctx.fillRect(translatedX - 4, translatedY, 12, 4); // Centered horizontally

  // Draw the vertical line of the cross (12 pixels long at 4x scale, centered on translatedY)
  ctx.fillRect(translatedX, translatedY - 4, 4, 12); // Centered vertically

  // Add double-click functionality to set a new cross location
  addCrossDoubleClickFunctionality(ctx, ctx.canvas, translatedX, translatedY, width, height, xStart, yStart);
}

//============================================================================
//============================================================================

// Add functionality to set the cross on double-click and use buttons for fine-tuning
function addCrossDoubleClickFunctionality(ctx, canvas, translatedX, translatedY, width, height, xStart, yStart) {
  let currentX = translatedX;
  let currentY = translatedY;

  // Get the popup content element to update dynamically
  const popupContent = document.getElementById('headerInfoContent');

  // Double-click event listener to set a new cross position
  canvas.addEventListener('dblclick', function (event) {
    const rect = canvas.getBoundingClientRect();
    const mouseX = event.clientX - rect.left;
    const mouseY = event.clientY - rect.top;

    // Set the new position for the cross
    currentX = Math.round(mouseX / 4) * 4; // Snap to the nearest 1x pixel (4x scale)
    currentY = Math.round(mouseY / 4) * 4; // Snap to the nearest 1x pixel (4x scale)

    // Clear the canvas and redraw the image from the off-screen canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);
	ctx.imageSmoothingEnabled = false; // Disable smoothing for pixel-perfect scaling
    ctx.drawImage(offScreenCanvas, 0, 0);//redraw image

    // Redraw the cross at the new position
    ctx.fillRect(currentX - 4, currentY, 12, 4); // Horizontal
    ctx.fillRect(currentX, currentY - 4, 4, 12); // Vertical

    // Convert the scaled coordinates back to the original scale for the header
    const newXStart = -Math.abs(currentX / 4); // Convert back to original 1x scale
    const newYStart = -Math.abs(currentY / 4); // Convert back to original 1x scale

    // Dynamically update the header info with the new coordinates
	console.clear(); //wipe all previous shit.
    document.querySelector('input[value="${headerInfo.xStart}"]').value = newXStart;
    document.querySelector('input[value="${headerInfo.yStart}"]').value = newYStart;
	
	// We now need to calculate the new xEnd and yEnd stuff....this is going to be skull numbing.
	//===============================================================================================
	//console.log('BoundRight:', boundingbox.boundRight, 'BoundBottom:', boundingbox.boundBottom);
	
	const newXend = boundingbox.boundRight - Math.abs(newXStart);
	const newYend = boundingbox.boundBottom - Math.abs(newYStart);
	document.querySelector('input[value="${headerInfo.xEnd}"]').value = newXend;
	document.querySelector('input[value="${headerInfo.yEnd}"]').value = newYend;
	//===============================================================================================

  });

//============================================================================
// Buttons for fine-tuning the position of the cross
//============================================================================

  document.getElementById('fineTuneXPlus').addEventListener('click', function () {
    currentX += 4; // Move the cross to the right by 1 pixel (4x scale)
    updateCanvasAndHeader(ctx, currentX, currentY, width, height, popupContent);
  });

  document.getElementById('fineTuneXMinus').addEventListener('click', function () {
    currentX -= 4; // Move the cross to the left by 1 pixel (4x scale)
    updateCanvasAndHeader(ctx, currentX, currentY, width, height, popupContent);
  });

  document.getElementById('fineTuneYPlus').addEventListener('click', function () {
    currentY += 4; // Move the cross down by 1 pixel (4x scale)
    updateCanvasAndHeader(ctx, currentX, currentY, width, height, popupContent);
  });

  document.getElementById('fineTuneYMinus').addEventListener('click', function () {
    currentY -= 4; // Move the cross up by 1 pixel (4x scale)
    updateCanvasAndHeader(ctx, currentX, currentY, width, height, popupContent);
  });
}

//============================================================================
//============================================================================

// Helper function to update the canvas and header after fine-tuning
function updateCanvasAndHeader(ctx, currentX, currentY, width, height, popupContent) {
  // Clear the canvas and redraw the image
  ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height);
  ctx.imageSmoothingEnabled = false; // Disable smoothing for pixel-perfect scaling
  ctx.drawImage(offScreenCanvas, 0, 0); //redraw image

  // Redraw the cross at the new fine-tuned position
  ctx.fillRect(currentX - 4, currentY, 12, 4); // Horizontal
  ctx.fillRect(currentX, currentY - 4, 4, 12); // Vertical

  // Convert the scaled coordinates back to the original scale for the header
  const newXStart = -Math.abs(currentX / 4); // Convert back to original 1x scale
  const newYStart = -Math.abs(currentY / 4); // Convert back to original 1x scale

  // Dynamically update the header info with the new coordinates
    document.querySelector('input[value="${headerInfo.xStart}"]').value = newXStart;
    document.querySelector('input[value="${headerInfo.yStart}"]').value = newYStart;
	
	const newXend = boundingbox.boundRight - Math.abs(newXStart);
	const newYend = boundingbox.boundBottom - Math.abs(newYStart);
	document.querySelector('input[value="${headerInfo.xEnd}"]').value = newXend;
	document.querySelector('input[value="${headerInfo.yEnd}"]').value = newYend;	
}

//============================================================================
//============================================================================

// Close the pop-up when the "Close" button is clicked
document.getElementById('closePopup').addEventListener('click', function() {
  document.getElementById('headerInfoPopup').style.display = 'none';
});