/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"

/*
 * Structure to hold a string.
 */
typedef struct quark
{
	/* Numeric equivalent */
	int num;

	/* String value */
	char *value;

	/* Next quark in list */
	struct quark *next;
} quark;

/*
 * List of quarks.
 */
static quark *quark_list;
static int num_quarks;

/*
 * Size of commander list.
 */
#define MAX_C_LIST    100

/*
 * List of commander info.
 */
commander c_list[MAX_C_LIST];
int num_commanders;


/*
 * Create a new quark, or return its number if it already exists.
 */
int create_quark(char *str)
{
	quark *q_ptr;

	/* Search for the string */
	for (q_ptr = quark_list; q_ptr; q_ptr = q_ptr->next)
	{
		/* Check this one */
		if (!strcmp(q_ptr->value, str))
		{
			/* Return value */
			return q_ptr->num;
		}
	}

	/* Create new quark */
	q_ptr = (quark *)malloc(sizeof(quark));

	/* Get number */
	q_ptr->num = num_quarks++;

	/* Save string */
	q_ptr->value = strdup(str);

	/* Insert into list */
	q_ptr->next = quark_list;
	quark_list = q_ptr;

	/* Return value */
	return q_ptr->num;
}

/*
 * Given a quark value, return the string.
 */
char *lookup_quark(int num)
{
	quark *q_ptr;

	/* Search for the value */
	for (q_ptr = quark_list; q_ptr; q_ptr = q_ptr->next)
	{
		/* Check this one */
		if (q_ptr->num == num)
		{
			/* Return the string */
			return q_ptr->value;
		}
	}

	/* Not found */
	return NULL;
}

/*
 * Read the list of random commanders from a file.
 */
static int read_commanders(FILE *fff)
{
	commander *c_ptr;
	char buf[1024], *ptr;
	int len, lineno = 0;
	int i, num, chance, minv, maxv;

	/* Read until the end of the file */
	while (fgets(buf, 1024, fff))
	{
		/* Count lines */
		lineno++;

		/* Get string length */
		len = (int)strlen(buf);

		/* Smash newlines */
		if (buf[len - 1] == '\n') buf[len - 1] = '\0';
		
		/* Skip empty lines */
		if (!buf[0]) continue;

		/* Skip comments */
		if (buf[0] == '#') continue;

		/* Check for correct format */
		if (buf[1] != ':')
		{
			/* Error */
			return lineno;
		}

		/* Determine what to do */
		switch (buf[0])
		{
			/* New entry */
			case 'N':
			{
				/* Check for too many */
				if (num_commanders == MAX_C_LIST)
				{
					/* Error */
					return lineno;
				}

				/* Get pointer */
				c_ptr = &c_list[num_commanders++];

				/* Get name */
				c_ptr->name = create_quark(buf + 2);

				/* Done */
				break;
			}

			/* Unit type */
			case 'T':
			{
				/* Get type */
				c_ptr->unit = create_quark(buf + 2);

				/* Done */
				break;
			}

			/* Bodyguards */
			case 'B':
			{
				/* Get min and max */
				sscanf(buf + 2, "%d:%d", &minv, &maxv);

				/* Save */
				c_ptr->min_body = minv;
				c_ptr->max_body = maxv;

				/* Advance past two colons */
				ptr = strtok(buf + 2, ":");
				ptr = strtok(NULL, ":");

				/* Check for illegal format */
				if (!ptr) return lineno;

				/* Get bodyguard type */
				c_ptr->name_body = create_quark(ptr);

				/* Done */
				break;
			}

			/* Squad */
			case 'U':
			{
				/* Get squad info */
				sscanf(buf + 2, "%d:%d:%d:%d", &num, &chance, &minv, &maxv);

				/* Save */
				c_ptr->squad_chance[num] = chance;
				c_ptr->squad_min[num] = minv;
				c_ptr->squad_max[num] = maxv;

				/* Skip three colons */
				ptr = strtok(buf + 2, ":");
				ptr = strtok(NULL, ":");
				ptr = strtok(NULL, ":");

				/* Check for illegal format */
				if (!ptr) return lineno;

				/* Get squad type */
				c_ptr->squad_name[num] = create_quark(ptr);

				/* Done */
				break;
			}

			/* Experience */
			case 'X':
			{
				/* Get min and max */
				sscanf(buf + 2, "%d:%d", &minv, &maxv);

				/* Save */
				c_ptr->min_xp = minv;
				c_ptr->max_xp = maxv;

				/* Done */
				break;
			}

			/* Magic */
			case 'M':
			{
				/* Skip past first colon */
				ptr = strtok(buf + 2, ":");

				/* Get each value */
				for (i = 0; i < 10; i++)
				{
					/* Check for illegal format */
					if (!ptr) return lineno;
					
					/* Scan value */
					sscanf(ptr, "%d", &num);
					
					/* Save */
					c_ptr->min_magic[i] = num;

					/* Advance pointer */
					ptr = strtok(NULL, ":");
				}

				/* Done */
				break;
			}

			/* Magic */
			case 'O':
			{
				/* Skip past first colon */
				ptr = strtok(buf + 2, ":");

				/* Get each value */
				for (i = 0; i < 10; i++)
				{
					/* Check for illegal format */
					if (!ptr) return lineno;
					
					/* Scan value */
					sscanf(ptr, "%d", &num);
					
					/* Save */
					c_ptr->max_magic[i] = num;

					/* Advance pointer */
					ptr = strtok(NULL, ":");
				}

				/* Done */
				break;
			}

			/* Random equipment */
			case 'R':
			{
				/* Get value */
				sscanf(buf + 2, "%d", &num);

				/* Save */
				c_ptr->rand_equip = num;

				/* Done */
				break;
			}

			/* Specific equipment */
			case 'I':
			{
				/* Get number and chance */
				sscanf(buf + 2, "%d:%d", &num, &chance);

				/* Save */
				c_ptr->item_chance[num] = chance;

				/* Skip one colon */
				ptr = strtok(buf + 2, ":");

				/* Check for illegal format */
				if (!ptr) return lineno;

				/* Get item name */
				c_ptr->item_name[num] = create_quark(ptr);

				/* Done */
				break;
			}

			/* Unknown line */
			default:
			{
				/* Return error */
				return lineno;
			}
		}
	}

	/* Success */
	return 0;
}
