/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * Modified by Bryon "LintMan" Daly (2004) for assorted improvements and bug fixes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"

/* Terrain type correspondence (hack) */
static int t_cor[TERRAIN_TYPE] = {
    TERRAIN_NONE,
    TERRAIN_NONE,
    TERRAIN_MOUNTAIN,
    TERRAIN_FARM,
    TERRAIN_FOREST,
    TERRAIN_SWAMP,
    TERRAIN_MOUNTAIN | TERRAIN_WASTE,
    TERRAIN_WASTE
};


/*
* Print #nostart lines if needed.
*
* We print these for provinces with too few neighbours, to prevent
* players from having poor starting positions.
*/
static void dump_nostart(FILE *fff)
{
    int i, j, k, n;

    /* Loop through provinces (fix: provs are numbered 1..num_provs) */
    for (i = 1; i <= num_provs; i++)
    {
        /* Assume no good neighbours */
        n = 0;

        /* Count neighbours of same type (land/sea) */
        for (j = 0; j < provs[i].num_neighbours; j++)
        {
            /* Get neighbour number */
            k = provs[i].neighbours[j];

            /* Skip neighbours of different type */
            if (provs[i].sea != provs[k].sea) continue;

            /* Count neighbour */
            n++;
        }

        /* Check for no good */
        if ((provs[i].sea && n < opt.min_neighbour_sea) ||
            (!provs[i].sea && n < opt.min_neighbour_land))
        {
            /* Don't let player start here */
            fprintf(fff, "#nostart %d\n", i);
        }
    }

    /* Blank line at end */
    fprintf(fff, "\n");
}

/*
* Dump terrain types.
*
* Very experimental.
*
* A Province is considered "terrain X" if at least 30% of its pixels are
* "terrain X".
* Exception is Sea, of course.
*/
static void dump_terrain(FILE *fff)
{
    int i,j, sea_nb, land_nb, ms_ct = 0;
    long total_pixels, total_lp_pixels = 0, total_sp_pixels = 0;
    int terrain;
    long avg_prov_pixels_land, avg_prov_pixels_sea;
    long small_prov_limit_land, small_prov_limit_sea, large_prov_limit_land, large_prov_limit_sea;
    /* keep running total of small/large to have statistics to report later */
    int large_prov_land = 0, large_prov_wat = 0, small_prov_land = 0, small_prov_wat = 0;
  
    
    // find total nums of land/sea pixels in land/sea provinces
    for (i=1; i <= num_provs; i++)
    {
        if (!provs[i].sea)
        {
            total_lp_pixels += provs[i].total_pixels;  // compute tot # pix in land provs
        } else
        {
            total_sp_pixels += provs[i].total_pixels;  // compute tot # pix in sea provs
        }
    }
    avg_prov_pixels_land = (long)((double)total_lp_pixels/opt.num_land);
    avg_prov_pixels_sea = (long)((double)total_sp_pixels/opt.num_sea);

    /*
    ** scale the averages to find upper/lower limits for a prov to be called a small/large prov
    ** This does not guarantee that provs will be evenly distributed as small/regular/large, since
    ** the actual province pixel sizes are random.  It does guarantee that "small" provinces will
    ** actually be smaller than average, and "large" provinces will be larger than average.
    ** A different method would be to just sort the provinces by size, and set the smallest x% as
    ** "small", and the largest y% as "large", which would guarantee the desired percentages of
    ** small/regular/large, but wouldn't guarantee that there was any significant visible size 
    ** differences between small and large provinces.
    */
    small_prov_limit_land = (long)(avg_prov_pixels_land * opt.small_prov_scal);
    small_prov_limit_sea = (long)(avg_prov_pixels_sea * opt.small_prov_scal);

    large_prov_limit_land = (long)(avg_prov_pixels_land * opt.large_prov_scal);
    large_prov_limit_sea = (long)(avg_prov_pixels_sea * opt.large_prov_scal);


    printf("small_prov_limit_land = %ld   small_prov_limit_sea = %ld\n", small_prov_limit_land, small_prov_limit_sea);
    printf("avg_prov_pixels_land = %ld   avg_prov_pixels_sea = %ld\n", avg_prov_pixels_land, avg_prov_pixels_sea);
    printf("large_prov_limit_land = %ld   large_prov_limit_sea = %ld\n", large_prov_limit_land, large_prov_limit_sea);

    for (i=1; i <= num_provs; i++)
    {
        terrain = 0;
        total_pixels = provs[i].total_pixels;

        // set small/large flags as appropriate, skip for blank maps
        if (!opt.blank_map)
        {
            if (provs[i].sea)
            {
                if (total_pixels >= large_prov_limit_sea)
                {
                    terrain = TERRAIN_LARGE;
                    large_prov_wat++;
                }
                else if (total_pixels <= small_prov_limit_sea)
                {
                    terrain = TERRAIN_SMALL;
                    small_prov_wat++;
                }
            } 
            else
            {
                if (total_pixels >= large_prov_limit_land)
                {
                    terrain = TERRAIN_LARGE;
                    large_prov_land++;
                }
                else if (total_pixels <= small_prov_limit_land)
                {
                    terrain = TERRAIN_SMALL;
                    small_prov_land++;
                }
            }
        }

        // Should this province have better chance of gem sites?  Skip for blanks
        if ((opt.amt_manysite > Urand()) && (!opt.blank_map))
        {
            terrain |= TERRAIN_MANYSITE;
            ms_ct++;
        }

        /* Compute terrain types (skip if making a blank, since PLAINS is default */
        if (!opt.blank_map)
        {
            for (j=1; j<TERRAIN_TYPE; j++)
            {
                /* 
                ** if province has 30% or more of a terrain type, add that terrain flag.
                ** But make sure we don't add land terrain types to sea provs 
                */
                if ((provs[i].pix[j] >= 0.3 * total_pixels) && (!provs[i].sea))
                {
                    terrain |= t_cor[j];
                }
            }
        }

        /* Is it a sea province? */
        if (provs[i].sea)
        {
            terrain |= TERRAIN_SEA;
        }

        /* Dump this province terrain */
        fprintf(fff, "#terrain %d %d", i, terrain);

        // Add a comment listing province land/sea type and its count of land/sea neighbors
        // in the form L-S2L3 for a land province with 2 sea and 3 land neighbors.
        fprintf(fff, "\t-- %s-", (provs[i].sea) ? "S" : "L");

        /* Loop through neighbors, counting land and sea connections */
        sea_nb = 0;
        land_nb = 0;

        // for each neighbor of province
        for (j = 0; j < provs[i].num_neighbours; j++)
        {
            // if neighbor is sea
            if (provs[provs[i].neighbours[j]].sea)
                sea_nb++;  // incr sea neighbors
            else
                land_nb++;  // else incr land neighbors
        }

        fprintf(fff, "S%dL%d\n", sea_nb, land_nb);
    }

    fprintf(fff, "\n-- Province statistics:\n");
    fprintf(fff, "-- Average land pixel size: %ld   Large prov limit: %ld   Small prov limit: %ld\n", 
        avg_prov_pixels_land, large_prov_limit_land, small_prov_limit_land);
    fprintf(fff, "-- Average sea pixel size: %ld   Large prov limit: %ld   Small prov limit: %ld\n", 
        avg_prov_pixels_sea, large_prov_limit_sea, small_prov_limit_sea);
    fprintf(fff, "-- Large provinces: %d land(%5.2f %%) / %d water(%5.2f %%)\n", 
        large_prov_land, (float)large_prov_land/opt.num_land*100, large_prov_wat, (float)large_prov_wat/opt.num_sea*100);
    fprintf(fff, "-- Small provinces: %d land(%5.2f %%) / %d water(%5.2f %%)\n", 
        small_prov_land, (float)small_prov_land/opt.num_land*100, small_prov_wat, (float)small_prov_wat/opt.num_sea*100);
    printf("-- Large provinces: %d land(%5.2f %%) / %d water(%5.2f %%)\n", 
        large_prov_land, (float)large_prov_land/opt.num_land*100, large_prov_wat, (float)large_prov_wat/opt.num_sea*100);
    printf("-- Small provinces: %d land(%5.2f %%) / %d water(%5.2f %%)\n", 
        small_prov_land, (float)small_prov_land/opt.num_land*100, small_prov_wat, (float)small_prov_wat/opt.num_sea*100);
    fprintf(fff, "-- Special magic sites: %d\n", ms_ct);
}


/*
* Dump province neighbour information.
*
* We only dump each association once.
*/
static void dump_neighbours(FILE *fff)
{
    int i, j, n = 0;

    /* 
    ** Loop through each province
    ** Note: provs are numbered 1..num_provs, but we can skip the last one 
    ** because its neighbors will already have been added
    */
    for (i = 1; i < num_provs; i++)
    {
        /* Loop through neighbours */
        for (j = 0; j < provs[i].num_neighbours; j++)
        {
            /* Skip neighbours below us */
            if (provs[i].neighbours[j] < i) continue;

            /* Dump this association */
            fprintf(fff, "#neighbour %d %d\n", i, provs[i].neighbours[j]);

            /* Count dumped associations */
            n++;
        }
    }

    /* Blank line at end */
    fprintf(fff, "\n");
}


/*
* Create a .map file with the given name.
*/
void dump_map(char *name, char *tga)
{
    FILE *fff;

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Generating %s file...\n", name);
    }

    /* Open map file */
    fff = fopen(name, "wb");

    /* Check for error */
    if (!fff)
    {
        /* Dump error */
        perror("fopen");
        exit(1);
    }

    /* Print the map title */
    fprintf(fff, "#dom2title %s\n", opt.map_name);
    fprintf(fff, "#imagefile %s\n", tga);
    fprintf(fff, "#domversion 2.02\n");
    fprintf(fff, "#defaultmapzoom 2.0\n");
    fprintf(fff, "#description \"Created with random map generator version %s.  ", VERSION);
    fprintf(fff, "%d land provinces, %d seas. ", opt.num_land, opt.num_sea);
    fprintf(fff, "Random seed: %d\"\n", opt.seed);
    fprintf(fff, "\n");

    /* Print #nostart flags if needed */
    dump_nostart(fff);
    /* Print province neighbours */
    dump_neighbours(fff);
    /* Print terrain types */
    dump_terrain(fff);


    /* Done */
    fclose(fff);
}
