/*
Selective Gaussian Blur
(c) Michael Cheng 2002  (web!planckCRAPenergy.com [to mail s/!/@/-CRAP])
Licensed under the LGPL

Heavily modified by Bryon "LintMan" Daly (2004) to work with DomMap data.  This 
removed the need for the selective "color distance" stuff, as the actual map data
is used to determine what should be blurred or not.

If you use it let me know.
If you improve it (very possible :), then also let me know. e.g. the brute force
method of applying the gaussian convolution filter.

Algorithm in-part based upon Axel Mellinger's bgsmooth (axm@rz.uni-potsdam.de)

This program implements a selective gaussian blur (as seen in the GIMP).
Only operates on ppm files at the moment. Include it in NetPBM?

A gaussian blur, blurs everything - losing detail.
A selective gaussian blur first checks to see whether the point near the
pivot point is within a certain colour distance, and if it isn't close
enough
then don't include it in the blur calculation.

Compile:
gcc selgauss.c -lppm -lm -o selgauss

Usage:
selgauss [inputfile] [radius] [threshold]

N.B. Radius isn't the size of the convolution matrix, it's the standard
deviation of the gaussian curve.  The actual size of the
convolution filter
is *larger* than this number.
radius      convolution size
1           5x5
2           9x9
3           13x13

Fix: Make the size of the NxN convolution matrix be the argument instead
of the
radius.  i.e. specify N=3,5,7,9,11.  More sensible to think about
for
the user.

Extension: Add finer distance controls.  E.g. Maybe only blur a
particular colour channel.
Or, maybe only calculate the distance in a subset of the
colour space
Or, weight the blurring of the different colour channels.

Fix: Gaussian convolution matrix is separable?  Can do the convolution
in 2N time instead of N^2?

History:
22 June 2001 - v0.1
First release.
*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>
#include "dommap.h"

// Arbitrary maximum size of the convolution matrix.
#define MAXCONV 50

#define RED(x)   (((x)>>16)&0xff)  // return red bits of RGB
#define GREEN(x) (((x)>>8)&0xff)   // return green bits of RGB
#define BLUE(x)  ((x)&0xff)        // return blue bits of RGB

#define MAKERGB(r,g,b)   ((((r)&0xff)<<16) | (((g)&0xff)<<8) | ((b)&0xff))

double matrix[MAXCONV][MAXCONV];
int ncnv;

BOOL get_pixel_terrain_color(int row, int col, int *r, int *g, int *b, BOOL sea_center);


void blur_terrain(void) 
{
    // selective gaussian blur.
    double ra=0.0, ga=0.0, ba=0.0; // Accumulators for each colour
    double norm=0.0; // Use to normalise the result of the accumulator.
    int r0, g0, b0, rgbcolor;
    int i,j;
    int row, col;
    int r, g, b;
    double val;
    BOOL center_water_pixel, blur_pixel, inc_pixel;
    
    matrix_init();

    for (row=0; row < opt.height; row++) 
    {
        for (col=0; col < opt.width; col++) 
        {
            norm=0.0; val=0.0; ra=0.0; ba=0.0; ga=0.0;
            center_water_pixel = (map[row][col] == WATER);
            blur_pixel = get_pixel_terrain_color(row, col, &r0, &g0, &b0, center_water_pixel);
            
            /* 
            ** Now for each pixel, convolve the conv_matrix with the pixels
            ** around it \
            */
            if (blur_pixel)
            {
                for (i=-ncnv; i<=ncnv; i++) 
                {
                    for (j =-ncnv; j<=ncnv; j++) 
                    {
                        /* Step through the convolve matrix */
                        val =  matrix[abs(i)][abs(j)];
                        inc_pixel = get_pixel_terrain_color(row+i, col+j, &r, &g, &b, center_water_pixel);
                        //if (colour_distance(r0,g0,b0,r,g,b) < (float)threshold)
                        if (inc_pixel)  // if we should include the pixel we just got into our blur matrix
                        {
                            norm += val;
                            ra += r * val;
                            ga += g * val;
                            ba += b * val;
                        }
                    }
                }
                rgbcolor = MAKERGB((int)(ra/norm + 0.5), (int)(ga/norm + 0.5), (int)(ba/norm + 0.5));

                if (rgbcolor == 0xFFFFFF)  // make sure new color isn't pure white, which is reserved for capitals
                    rgbcolor = 0xFEFEFE;  // just a slightly darker white
            } else // don't blur this pixel
            {
                rgbcolor = color_map[map[row][col]];  // index color map with terrain type
            }
            
            blur_map[row][col] = rgbcolor;  // build new blurred map version
        }
    }
}


/*
** Get the rgb values for a pixel based on its terrain map.  Does checking to make 
** sure the blur should affect (or be affected by) this pixel.  sea_center is
** set when the pixel being blurred is water, and cleared when it is not water - this
** allows use to enable/disable land/sea cross-blurring from the surrounding pixels.
** If the pixel is invalid (ie: off the map, or a border/capital, etc, returns FALSE.
*/
BOOL get_pixel_terrain_color(int row, int col, int *r, int *g, int *b, BOOL sea_center)
{
    int terrain_type, color;
    *r = -1;
    *g = -1;
    *b = -1;

    if ((row < 0) || (row >= opt.height))
        return (FALSE);  // off the map
    if ((col < 0) || (col >= opt.width))
        return (FALSE);  // off the map

    // figure out the terrain type and color
    terrain_type = map[row][col];
    color = color_map[terrain_type];

    *r = RED(color);
    *g = GREEN(color);
    *b = BLUE(color);

    if ((!opt.blur_sea) && (terrain_type == WATER))
    {
        return (FALSE);  // don't touch any sea pixels if !opt.blur_sea
    }

    // if land/sea cross-blurring is disabled, check to see if blur target 
    // pixel is land and this pixel is sea (or vice-versa)
    if ((!opt.blur_sea) && (!opt.blur_land_sea) && (((sea_center) && 
        (terrain_type != WATER)) || ((!sea_center) && (terrain_type == WATER))))
    {
        return (FALSE);  // this is a cross-over pixel, don't blur in if option not set
    }

    if ((terrain_type == LAND_BORDER) || (terrain_type == SEA_BORDER) || (terrain_type == CAPITAL) || (terrain_type == CAP_BORDER))
    {
        return (FALSE);  // don't blur borders or capital
    }

    return (TRUE);  // OK to blur with this pixel
}


void aa_borders(void)
{
    double rnew=0.0, gnew=0.0, bnew=0.0; 
    double r0, g0, b0;
    int row, col;
    double bc_red, bc_green, bc_blue;
    double blur_scal;
    int terrain_type, tcolor, rgbcolor;
    int y, x, dir;

    for (row=0; row < opt.height; row++) 
    {
        for (col=0; col < opt.width; col++) 
        {
            terrain_type = map[row][col];
            // if we find a border pixel, push some shading to all surrounding pixels
            if ((terrain_type == LAND_BORDER) || (terrain_type == SEA_BORDER))
            {
                bc_red = RED(color_map[terrain_type]);
                bc_green = GREEN(color_map[terrain_type]);
                bc_blue = BLUE(color_map[terrain_type]);

                /* Spread each direction, including diags */
                for (dir = 0; dir < 8; dir++)
                {
                    /* Get new grid location */
                    y = row + dy[dir];
                    x = col + dx[dir];

                    /* Check for out of bounds */
                    if ((y < 0) || (y >= opt.height)) continue;
                    if ((x < 0) || (x >= opt.width)) continue;

                    terrain_type = map[y][x];
                    
                    // leave surrounding borders alone
                    if ((terrain_type == LAND_BORDER) || (terrain_type == SEA_BORDER))
                        continue;
                    
                    /*
                    ** set the blur scalar that scales down the effect of averaging in
                    ** the border color.  This number should be small because we might
                    ** be adding it multiple times if the pixel is adjacent to several
                    ** border pixels.  And if we're on a diag, use an even smaller diag 
                    ** blur scalar.
                    */
                    blur_scal = (dir <= 3) ? opt.border_blur_scal : opt.border_blur_diag;

                    // get already-blurred terrain color for this pixel
                    tcolor = blur_map[y][x];
                    
                    r0 = (double)RED(tcolor);
                    g0 = (double)GREEN(tcolor);
                    b0 = (double)BLUE(tcolor);

                    // average the scaled-down border color into the terrain color
                    rnew = (r0 + bc_red * blur_scal)/(1+blur_scal) + 0.5;
                    gnew = (g0 + bc_green * blur_scal)/(1+blur_scal) + 0.5;
                    bnew = (b0 + bc_blue * blur_scal)/(1+blur_scal) + 0.5;

                    // make sure we don't end up with any colors > 255
                    if (rnew >= 255) 
                        rnew = 255.0;
                    if (gnew >= 255.0) 
                        gnew = 255.0;
                    if (bnew >= 255.0)
                        bnew = 255.0;
                    
                    rgbcolor = MAKERGB((int)rnew, (int)gnew, (int)bnew);

                    if (rgbcolor == 0xffffff)  // if we somehow ended up with pure white
                        rgbcolor = 0xfffefe;  // make it not pure white anymore
                    
                    // replace old color with new one
                    blur_map[y][x] = rgbcolor;
                }
            }
        }
    }
}


void matrix_init() 
{
    double r, rclip;               /* clip where Gaussian has dropped to 0.1 */
    int i,j;
    int maxcnv = MAXCONV;
    //double val, sum = 0;

    rclip =  (double)opt.blur_radius*sqrt(-log(0.1)/log(2.0));
    //fprintf(stderr, "rclip: %f radius: %i\n", rclip, opt.blur_radius);
    ncnv = (int)rclip + 1;
    //fprintf(stderr,"size: ncnv %i\n",ncnv);
    if (ncnv>maxcnv) {
        printf("\nError: radius too big\n");
        exit(1);
    }

    for (i=0; i<=ncnv; i++) 
    {
        for (j=0; j<=ncnv; j++) 
        {
            r = sqrt(i*i+j*j);
            matrix[i][j] = exp( -log(2)*pow((r/opt.blur_radius),2) );
            //printf("%5.3f ", matrix[i][j]);
        }
        //printf("\n");
    }

    /*
    // print out the whole matrix
    for (i=-ncnv; i<=ncnv; i++) 
    {
        for (j =-ncnv; j<=ncnv; j++) 
        {
            // Step through the convolve matrix 
            val =  matrix[abs(i)][abs(j)];
            sum += val;
            printf("%5.3f ", val);
        }
        printf("\n");
    }
    printf("sum = %5.3f\n", sum);
    */
}

