/*
* Dominions random map generator.
*
* Copyright (c) 2002 Keldon Jones
* 
* Modified by Philippe Duchon (2004) for compatibility with Dominions 2
*
* Modified by Bryon "LintMan" Daly (2004) for assorted improvements and bug fixes
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

/* Version number */
#define VERSION        "2.17"

/* For Windows compatibility? */
#define M_PI 3.14159265358979323846

/* Terrain types */
#define PLAINS         0
#define WATER          1
#define MOUNTAIN       2
#define FARMLAND       3
#define FOREST         4
#define SWAMP          5
#define TUNDRA         6
#define WASTELAND      7
#define LAND_BORDER    8
#define SEA_BORDER     9
#define CAPITAL        10
// we should only use CAP_BORDER at the very end to make the .tga 
// unless we go through the rest of the code to make sure it doesn't
// barf on this pseudo "terrain type"
#define CAP_BORDER     11  

#define TERRAIN_TYPE 8  // number of types above that are actual terrain
#define TOTAL_TERRAIN_TYPES  (TERRAIN_TYPE + 4)  // all possible types

/* Dominions 2 terrain codes for the map (to be ORed) */
#define TERRAIN_NONE     0
#define TERRAIN_SMALL    0x00000001
#define TERRAIN_LARGE    0x00000002
#define TERRAIN_SEA      0x00000004
#define TERRAIN_RIVER    0x00000008
#define TERRAIN_MOUNTAIN 0x00000010
#define TERRAIN_SWAMP    0x00000020
#define TERRAIN_WASTE    0x00000040
#define TERRAIN_FOREST   0x00000080
#define TERRAIN_FARM     0x00000100
#define TERRAIN_NOSTART  0x00000200
#define TERRAIN_MANYSITE 0x00000400

/* Province types */
#define PROV_CIRC      0
#define PROV_VERT      1
#define PROV_HORZ      2
#define PROV_BIG       3
#define PROV_SMALL     4

/* Various limits */
#define MAX_SQUADS     5
#define MAX_EQUIP      10
#define MAX_SITES      4
#define MAX_COMMANDERS 10


/* The code is straight C, no boolean type, so make one up - BKD */
#define TRUE    1
#define FALSE   0
typedef int BOOL;

// for easier readability than the 0's and 1's being used now in places
#define LAND    0
#define SEA     1

#define SWITCHSTR(x) ((x) ? "ON" : "OFF")

/*
* Information about a province.
*/
typedef struct province
{
    /* Location of the capital pixel */
    int cap_y;
    int cap_x;

    /* True if sea province (or use LAND/SEA #defines)*/
    int sea;

    /* Type (shape) of province */
    int type;

    /* Number of pixels of each type */
    int pix[TERRAIN_TYPE];

    /* total number of pixels in province */
    int total_pixels;

    /* List of neighbours */
    int *neighbours;

    /* Number of neighbours */
    int num_neighbours;
    
    /* List of deleted neighbours - so we can add them back if necessary */
    int *del_nb;

    /* Number of deleted neighbours */
    int num_del_nb;

    /* track whether this province is connected to the main graph */
    BOOL connected;
} province;

/*
* Structure containing options.
*/
typedef struct options
{
    /* Verbosity */
    BOOL verbose;

    /* Size of map */
    int height;
    int width;

    /* Random seed */
    int seed;

    /* Terrain ruggedness */
    double rugged;

    /* Elevation distribution */
    double water_level;
    double tree_level;
    double mountain_level;

    /* Richness distribution */
    double poor_level;
    double rich_level;

    /* Terrain dithering */
    double terrain_dither;

    /* Number of provinces */
    int num_sea;
    int num_land;

    /* Percentage of special provinces */
    int amt_special;

    /* Amount of shore to avoid */
    int shore_avoidance_sea;
    int shore_avoidance_land;

    /* Amount of spread between capitals */
    int sea_spread;
    int land_spread;

    /* Cost of province spanning sea */
    int sea_cost;

    /* Cost of province spanning differing terrain */
    int terrain_cost;

    /* Amount of border smoothing */
    int smooth;

    /* Minimum neighbours needed to be starting province */
    int min_neighbour_sea;
    int min_neighbour_land;

    /* Name of map */
    char *map_name;

    /* Colours */
    long plains_color;
    long water_color;
    long mountain_color;
    long farmland_color;
    long forest_color;
    long swamp_color;
    long tundra_color;
    long wasteland_color;
    long sea_border_color;
    long land_border_color;
    long capital_color;
    long cap_border_color;  /* color to use for borders around capital */

    BOOL cap_border;  /* draw borders around capital? */

    /* Shading options */
    BOOL do_shade;
    BOOL do_shade_water;
    double shade_ruggedness;  // let shade have a separate rugged val so very rugged maps don't get wacked shading
    int elevation_brightness_factor;
    int elevation_shade_factor;
    int shadows_brightness_factor;
    int shadows_shade_factor;

    /* uncrossable border options */
    double impass;     /* fraction impassable borders (0.0-1.0) */
    BOOL triple_thick;  /* 1 = uncrossable borders drawn triple thickness, 0 - double thickness */
    BOOL sea_sea_impass;  /* 1 = allow impassable borders between sea provs, 0 = not allowed */

    BOOL no_force_water;   /* don't force all land bits in the sea provs to be water pixels */

    double small_prov_scal;  /* fraction of the average province size to be considered "small" (default: 0.55) */
    double large_prov_scal;  /* multiple of the average province size to be considered "large" (default: 1.25) */

    double amt_manysite;  /* fraction of provinces that are marked for having many sites (0.0-1.0)*/

    BOOL blank_map;  /* produce a "blank" map: no (visible) terrain except sea, plains */
    
    /* terrain blur options */
    int blur_radius;  /* this is not radius in pixels, 0 = no blurring */
    BOOL blur_land_sea;  /* blur land into sea & sea into land? */
    BOOL blur_sea; /* blur the sea at all (it's a solid color; shadows added later) */
    double border_blur_scal;  /* amount to weight border color in adj. pixels */
    double border_blur_diag;  /* amount to weight border color in adj. diag pixels */
} options;


/*
* Information about a random commander.
*/
typedef struct commander
{
    /* Name */
    int name;

    /* Type of unit */
    int unit;

    /* Bodyguards */
    int min_body;
    int max_body;
    int name_body;

    /* Squads */
    int squad_chance[MAX_SQUADS];
    int squad_min[MAX_SQUADS];
    int squad_max[MAX_SQUADS];
    int squad_name[MAX_SQUADS];

    /* Experience */
    int min_xp;
    int max_xp;

    /* Magic values */
    int min_magic[10];
    int max_magic[10];

    /* Random equipment */
    int rand_equip;

    /* Specific equipment */
    int item_chance[MAX_EQUIP];
    int item_name[MAX_EQUIP];

} commander;

/*
* Information about a random independent province.
*/
typedef struct independent
{
    /* Name */
    int name;

    /* Rarity */
    int rarity;

    /* Magic sites */
    int site_chance[MAX_SITES];
    int site_name[MAX_SITES];

    /* Fort */
    int fort_chance;
    int fort_type;

    /* Lab and temple */
    int temple_chance;
    int lab_chance;

    /* Population and unrest */
    int min_pop;
    int max_pop;
    int min_unrest;
    int max_unrest;
    int pop_type;

    /* Commanders */
    int commanders[MAX_COMMANDERS];

    /* Flags */
    int flags;

} independent;


/* Function prototypes */
extern void generate_map();
extern void create_provinces();
extern void dump_tga(char *name);
extern void dump_map(char *name, char *tga);
extern int dominions_neighbours(int p1, int p2);
extern double Urand();
void init_genrand(unsigned long s);  // In mt19937ar.c, Mersenne Twister random number init
double genrand_real1(void);  // In mt19937ar.c, Mersenne Twister random number gen [0,1) range
void matrix_init(void);
void blur_terrain(void);
void aa_borders(void);


/* External variables */
extern options opt;
extern int **map;
extern int **blur_map;
extern double **shadow;
extern double **elevation;
extern double min_shadow;
extern double max_shadow;
extern double min_elevation;
extern double max_elevation;
extern province *provs;
extern int num_provs;
extern long color_map[];
extern int dy[8];
extern int dx[8];