/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * Modified by Bryon "LintMan" Daly (2004) for assorted improvements and bug fixes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"

/*
* Global options
*/
options opt;

/*
* Array containing map pixels.
*/
int **map;

/*
* Array containing elevation pixels
*/
double **elevation;
double **shadow;
double min_shadow = 0;
double max_shadow = 0;
double min_elevation = 0;
double max_elevation = 0;

/*
* Create map array.
*/
static void make_map(void)
{
    int y;

    /* Make array of rows */
    //map = (int **)malloc(sizeof(int *) * opt.height);
    map = (int **)calloc(opt.height, sizeof(int *));

    /* Make each row */
    for (y = 0; y < opt.height; y++)
    {
        /* Make one row */
        //map[y] = (int *)malloc(sizeof(int *) * opt.width);
        map[y] = (int *)calloc(opt.width, sizeof(int *));
    }
}

/*
* Set options to the default.
*/
static void set_default_options(void)
{
    /* Quiet mode */
    opt.verbose = FALSE;

    /* Map size */
    opt.height = 550;
    opt.width = 750;

    /* Random seed */
    opt.seed = 0;

    /* Ruggedness */
    opt.rugged = 1.5;

    /* Elevation distribution */
    opt.water_level = 35.0;
    opt.tree_level = 60.0;
    opt.mountain_level = 80.0;

    /* Richness distribution */
    opt.poor_level = 20.0;
    opt.rich_level = 80.0;

    /* Terrain dithering */
    opt.terrain_dither = 0.08;

    /* Number of provinces */
    opt.num_sea = 10;
    opt.num_land = 90;

    /* Amount of special shape provinces */
    opt.amt_special = 20;

    /* Shore avoidance */
    //opt.shore_avoidance_sea = 30;
    opt.shore_avoidance_sea = 8;  // makes lakes and small sea provs much likelier
    opt.shore_avoidance_land = 8;

    /* Capital spread */
    //opt.sea_spread = 60;
    opt.sea_spread = 80;  // helps make lakes and small sea provs much likelier
    //opt.land_spread = 25;
    opt.land_spread = 40;  // helps make peninsula and island land provs much likelier

    /* Shore-crossing cost */
    opt.sea_cost = 1000000;

    /* Terrain crossing cost */
    opt.terrain_cost = 10;

    /* Some smoothing */
    opt.smooth = 1;

    /* Min neighbours needed */
    opt.min_neighbour_sea = 2;
    opt.min_neighbour_land = 4;

    /* Map name */
    opt.map_name = "random";

    /* Colors */
    /* // original default colors
	opt.plains_color = 0xC58F54;
	opt.water_color = 0x4E7283;
	opt.mountain_color = 0x5B4524;
	opt.farmland_color = 0x708841;
	opt.forest_color = 0x308020;
	opt.swamp_color = 0x34969F;
	opt.tundra_color= 0xC4B0A4;
	opt.wasteland_color = 0xAD9374;
	opt.land_border_color = 0xA00000;
	opt.sea_border_color = 0x2C87C3;
	opt.capital_color = 0xFFFFFF;
    */
    // new default colors    
    opt.plains_color = 0x33AA11; //0xC58F54;
    opt.water_color = 0x44AADD; //0x4E7283;
    opt.mountain_color = 0x594020; //0x5B4524;
    opt.farmland_color = 0x55BB55; //0x708841;
    opt.forest_color = 0x337733; //0x308020;
    opt.swamp_color = 0x55AA88; //0x34969F;
    opt.tundra_color= 0x707070; //0xC4B0A4;
    opt.wasteland_color = 0xA0522D; //0xAD9374;
    opt.land_border_color = 0xb43c00;//0xFF0000; //0xA00000;
    //opt.land_border_color = 0xFF0000; //0xA00000;
    opt.sea_border_color = 0x551144; //0x2C87C3;
    opt.capital_color = 0xFFFFFF;
    opt.cap_border_color = 0xFFFF00;  /* color to use for borders around capital */

    opt.cap_border = TRUE;  /* draw borders around capital? */
     
    /* Shading */
    opt.do_shade = TRUE;
    opt.do_shade_water = TRUE;
    opt.shade_ruggedness = 1.5;
    opt.elevation_brightness_factor = 32;
    opt.elevation_shade_factor = 32;
    opt.shadows_brightness_factor = 96;
    opt.shadows_shade_factor = 128;

    opt.impass = 0.15;      /* 15 % impassable borders default */
    opt.triple_thick = FALSE;   /* default double thickness impassable borders */
    opt.sea_sea_impass = FALSE;  /* 1 = allow impassable borders between sea provs, 0 = not allowed */

    opt.no_force_water = TRUE;  /* don't force all land bits in the sea provs to be water pixels */

    opt.small_prov_scal = 0.65;  /* fraction of the average province size to be considered "small" (default: 0.65) */
    opt.large_prov_scal = 1.25;  /* multiple of the average province size to be considered "large" (default: 1.25) */

    opt.amt_manysite = 0.10;  /* 10% of provs are marked manysite by default */

    opt.blank_map = FALSE;  /* default: make regular map (1 = make blank map) */

    /* terrain blur options */
    opt.blur_radius = 1;  /* this is not radius in pixels, 0 = no blurring */
    opt.blur_land_sea = TRUE;  /* blur land into sea & sea into land? */
    opt.blur_sea = TRUE; /* blur the sea at all? (it's a solid color; shadows added later) */
    opt.border_blur_scal = 0.5;  /* amount to weight border color in adj. pixels */
    opt.border_blur_diag = 0.25;  /* amount to weight border color in adj. diag pixels */
}

/*
* Print command-line help.
*/
static void print_help(int argc, char *argv[])
{
    /* Print help */
    fprintf(stderr, "Usage: %s <options>\n", argv[0]);
    fprintf(stderr, "  -v (for verbose mode) default: %s\n", SWITCHSTR(opt.verbose));
    fprintf(stderr, "  -help (print this option list)\n");
    fprintf(stderr, "  -h <map height> (100-2400) default: %d\n", opt.height);
    fprintf(stderr, "  -w <map width> (100-3200) default: %d\n", opt.width);
    fprintf(stderr, "  -n <map name> default: '%s'\n", opt.map_name);
    fprintf(stderr, "  -s <random seed>  default: timeseed\n");
    fprintf(stderr, "  -r <terrain ruggedness> default: %5.2f\n", opt.rugged);
    fprintf(stderr, "  -sr <Shadow map ruggedness> default: %5.2f\n", opt.shade_ruggedness);
    fprintf(stderr, "  -wl <water line (0-100)> default: %5.2f\n", opt.water_level);
    fprintf(stderr, "  -tl <tree line (0-100)> default: %5.2f\n", opt.tree_level);
    fprintf(stderr, "  -ml <mountain line (0-100)> default: %5.2f\n", opt.mountain_level);
    fprintf(stderr, "  -pl <poverty line (0-100)> default: %5.2f\n", opt.poor_level);
    fprintf(stderr, "  -rl <rich line (0-100)> default: %5.2f\n", opt.rich_level);
    fprintf(stderr, "  -d <terrain dithering> default: %5.3f\n", opt.terrain_dither);
    fprintf(stderr, "  -ns <num of sea provinces> default: %d\n", opt.num_land);
    fprintf(stderr, "  -nl <num of land provinces> default: %d\n", opt.num_sea);
    fprintf(stderr, "  -ps <percent special shape provinces> default: %d\n", opt.amt_special);
    fprintf(stderr, "  -sa <min distance from shore (sea)> default: %d\n", opt.shore_avoidance_sea);
    fprintf(stderr, "  -la <min distance from shore (land)> default: %d\n", opt.shore_avoidance_land);
    fprintf(stderr, "  -ss <distance between sea provinces> default: %d\n", opt.sea_spread);
    fprintf(stderr, "  -ls <distance between land provinces> default: %d\n", opt.land_spread);
    fprintf(stderr, "  -sc <cost for spreading over shore> default: %ld\n", opt.sea_cost);
    fprintf(stderr, "  -tc <cost for spreading over terrain> default: %d\n", opt.terrain_cost);
    fprintf(stderr, "  -sm <amount of border smoothing> default: %d\n", opt.smooth);
    fprintf(stderr, "  -ln <minimum neighbours for starting (land)> default: %d\n", opt.min_neighbour_land);
    fprintf(stderr, "  -sn <minimum neighbours for starting (sea)> default: %d\n", opt.min_neighbour_sea);
    fprintf(stderr, "  -imp <percentage impassable borders (0-100)> default: %5.2f\n", opt.impass*100);
    fprintf(stderr, "  -th3 (use triple-thickness for impassable borders) default: %s\n", SWITCHSTR(opt.triple_thick));
    fprintf(stderr, "  -th2 (use double-thickness for impassable borders) default: %s\n", SWITCHSTR(!opt.triple_thick));
    fprintf(stderr, "  -ssi <1 = allow sea-sea impassable borders, 0 = not allowed> default: %s\n", SWITCHSTR(opt.sea_sea_impass));
    fprintf(stderr, "  -ms <percent provinces marked as ManySite (0-100)> default: %5.2f\n", opt.amt_manysite*100.0);
    fprintf(stderr, "  -nfw (do not force all water pixel in sea provinces) default: %s\n", SWITCHSTR(opt.no_force_water));
    fprintf(stderr, "  -fw (force all water pixels in sea provinces) default: %s\n", SWITCHSTR(!opt.no_force_water));
    fprintf(stderr, "  -ba <on|off> (Turns additional shading on or off) default: %s\n", SWITCHSTR(opt.do_shade));
    fprintf(stderr, "  -bw <on|off> (Turns additional shading of water on or off) default: %s\n", SWITCHSTR(opt.do_shade_water));
    fprintf(stderr, "  -beb <0-255> Max amount to brighten a color due to elevation-map default: %d\n", opt.elevation_brightness_factor);
    fprintf(stderr, "  -bes <0-255> Max amount to shade a color due to elevation default: %d\n", opt.elevation_shade_factor);
    fprintf(stderr, "  -bsb <0-255> Max amount to brighten a color due to shadow-map default: %d\n", opt.shadows_brightness_factor);
    fprintf(stderr, "  -bss <0-255> Max amount to shade a color due to shadow-map default: %d\n", opt.shadows_shade_factor);
    fprintf(stderr, "  -c_plains <0xrrggbb plains color RGB value>  default: 0x%6X\n", opt.plains_color);
    fprintf(stderr, "  -c_water <0xrrggbb water color RGB value>  default: 0x%6X\n", opt.water_color);
    fprintf(stderr, "  -c_mountain <0xrrggbb mountain color RGB value>  default: 0x%6X\n", opt.mountain_color);
    fprintf(stderr, "  -c_farmland <0xrrggbb farmland color RGB value>  default: 0x%6X\n", opt.farmland_color);
    fprintf(stderr, "  -c_forest <0xrrggbb forest color RGB value>  default: 0x%6X\n", opt.forest_color);
    fprintf(stderr, "  -c_swamp <0xrrggbb swamp color RGB value>  default: 0x%6X\n", opt.swamp_color);
    fprintf(stderr, "  -c_tundra <0xrrggbb tundra color RGB value>  default: 0x%6X\n", opt.tundra_color);
    fprintf(stderr, "  -c_wasteland <0xrrggbb wasteland color RGB value>  default: 0x%6X\n", opt.wasteland_color);
    fprintf(stderr, "  -c_land_border <0xrrggbb land border color RGB value>  default: 0x%6X\n", opt.land_border_color);
    fprintf(stderr, "  -c_sea_border <0xrrggbb sea border color RGB value>  default: 0x%6X\n", opt.sea_border_color);
    fprintf(stderr, "  -c_cap_border <0xrrggbb capital border color RGB value>  default: 0x%6X\n", opt.cap_border_color);
    fprintf(stderr, "  -cb <on|off> (Turns capital borders on or off) default: %s\n", SWITCHSTR(opt.cap_border));
    fprintf(stderr, "  -sps <max %% of avg prov size to be a small province (0-100) default: %5.2f\n", opt.small_prov_scal*100.0);
    fprintf(stderr, "  -lps <min %% of avg prov size to be a large province (100+) default: %5.2f\n", opt.large_prov_scal*100.0);
    fprintf(stderr, "  -aa <scale border aa effect> default: %5.2f\n", opt.border_blur_scal);
    fprintf(stderr, "  -aad <scale border aa effect on diagonals> default: %5.2f\n", opt.border_blur_diag);
    fprintf(stderr, "  -blank (produce a 'blank' map) default: %s\n", SWITCHSTR(opt.blank_map));
    fprintf(stderr, "  -gbr <gaussian blur radius: integer 0..10> default %d\n", opt.blur_radius);
    /* Done */
    exit(1);
}

/*
* Parse command line options.
*/
static void get_command_line_options(int argc, char *argv[])
{
    int i;

    /* Parse each option */
    for (i = 1; i < argc; i++)
    {
        /* Check this option */
        if (!strcmp(argv[i], "-v"))
        {
            /* Set verbose mode */
            opt.verbose = 1;
        }
        else if(!strcmp(argv[i], "-help"))
        {
            print_help(argc, argv);
        } 
        else if (!strcmp(argv[i], "-h"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -h\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set height */
            opt.height = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-w"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -w\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set width */
            opt.width = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-n"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -n\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set name */
            opt.map_name = argv[++i];
        }
        else if (!strcmp(argv[i], "-s"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -s\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set seed */
            opt.seed = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-r"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -r\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set ruggedness */
            opt.rugged = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-sr"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -sr\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set ruggedness */
            opt.shade_ruggedness = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-wl"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -wl\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set water line */
            opt.water_level = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-tl"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -tl\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set tree line */
            opt.tree_level = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-ml"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ml\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set mountain line */
            opt.mountain_level = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-pl"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -pl\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set poverty line */
            opt.poor_level = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-rl"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -rl\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set rich line */
            opt.rich_level = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-d"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -d\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set dithering */
            opt.terrain_dither = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-ns"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ns\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set sea provinces */
            opt.num_sea = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-nl"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -nl\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set land provinces */
            opt.num_land = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-ps"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ps\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set percent special provinces */
            opt.amt_special = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-sa"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -sa\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set sea shore avoidance */
            opt.shore_avoidance_sea = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-la"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -la\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set land shore avoidance */
            opt.shore_avoidance_land = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-ss"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ss\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set sea spread */
            opt.sea_spread = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-ls"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ls\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set land spread */
            opt.land_spread = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-sc"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -sc\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set shore cost */
            opt.sea_cost = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-tc"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -tc\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set shore cost */
            opt.terrain_cost = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-sm"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -sm\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set smoothing amount */
            opt.smooth = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-sn"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -sn\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set min neighbours */
            opt.min_neighbour_sea = atoi(argv[++i]);
        }
        else if(!strcmp(argv[i], "-ba"))
        {
           char* onOff;
           if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ba\n");
                print_help(argc, argv);
                exit(1);
            }
            onOff = argv[++i];
            if( strcmp( onOff, "on" ) == 0 ) {
                opt.do_shade = 1;
            }
            else if ( strcmp( onOff, "off" ) == 0 ) {
                opt.do_shade = 0;
            }
            else {
                print_help( argc, argv );
            }
        } 
        else if(!strcmp(argv[i], "-bw"))
        {
           char* onOff;
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -bw\n");
                print_help(argc, argv);
                exit(1);
            }
            onOff = argv[++i];
            if( strcmp( onOff, "on" ) == 0 ) {
                opt.do_shade_water = 1;
            }
            else if ( strcmp( onOff, "off" ) == 0 ) {
                opt.do_shade_water = 0;
            }
            else {
                print_help( argc, argv );
            }
        } 
        else if(!strcmp(argv[i], "-beb"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -beb\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.elevation_brightness_factor = atoi(argv[++i]);
        }
        else if(!strcmp(argv[i], "-bes")) 
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -bes\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.elevation_shade_factor = atoi(argv[++i]);
        }
        else if(!strcmp(argv[i], "-bsb")) 
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -bsb\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.shadows_brightness_factor = atoi(argv[++i]);
        }
        else if(!strcmp(argv[i], "-bss")) 
        {		    
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -bss\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.shadows_shade_factor = atoi(argv[++i]);
        }
        else if(!strcmp(argv[i], "-c_plains"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_plains\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.plains_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_water"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_water\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.water_color = strtol(argv[++i], NULL, 16);
        } 
        else if(!strcmp(argv[i], "-c_mountain"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_mountain\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.mountain_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_farmland"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_farmland\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.farmland_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_forest"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_forest\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.forest_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_swamp"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_swamp\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.swamp_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_tundra"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_tundra\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.tundra_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_wasteland"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_wasteland\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.wasteland_color = strtol(argv[++i], NULL, 16);;
        } 
        else if(!strcmp(argv[i], "-c_land_border"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_land_border\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.land_border_color = strtol(argv[++i], NULL, 16);
        } 
        else if(!strcmp(argv[i], "-c_cap_border"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_cap_border\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.cap_border_color = strtol(argv[++i], NULL, 16);
        } 
        else if(!strcmp(argv[i], "-c_sea_border"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -c_sea_border\n");
                print_help(argc, argv);
                exit(1);
            }
            opt.sea_border_color = strtol(argv[++i], NULL, 16);
        } 
        else if(!strcmp(argv[i], "-cb"))
        {
           char* onOff;
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -cb\n");
                print_help(argc, argv);
                exit(1);
            }
            onOff = argv[++i];
            if( strcmp( onOff, "on" ) == 0 ) {
                opt.cap_border = 1;
            }
            else if ( strcmp( onOff, "off" ) == 0 ) {
                opt.cap_border = 0;
            }
            else {
                print_help( argc, argv );
            }
        } 
        else if (!strcmp(argv[i], "-ln"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ln\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set min neighbours */
            opt.min_neighbour_land = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-imp"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -imp\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set impassable border fraction */
            opt.impass = (atof(argv[++i])/100);  // convert from percentage to fraction
        }
        else if (!strcmp(argv[i], "-th3"))
        {
            /* Triple thickness impassable borders */
            opt.triple_thick = 1;
        }
        else if (!strcmp(argv[i], "-th2"))
        {
            /* Double thickness impassable borders */
            opt.triple_thick = 0;
        }
        else if (!strcmp(argv[i], "-ssi"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ssi\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set sea-sea impassable border option */
            opt.sea_sea_impass = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-nfw"))
        {
            /* do not use force_water() routine */
            opt.no_force_water = 1;
        }
        else if (!strcmp(argv[i], "-fw"))
        {
            /* use force_water() routine if smoothing is on */
            opt.no_force_water = 0;
        }
        else if (!strcmp(argv[i], "-sps"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -sps\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set small province scalar */
            opt.small_prov_scal = (atof(argv[++i])/100.0);  // convert from %
        }
        else if (!strcmp(argv[i], "-lps"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -lps\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set large province scalar */
            opt.large_prov_scal = (atof(argv[++i])/100.0);  // convert from %
        }
        else if (!strcmp(argv[i], "-ms"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -ms\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set manysite percentage */
            opt.amt_manysite = (atof(argv[++i])/100.0);  // convert percentage to fraction
        }
        else if (!strcmp(argv[i], "-blank"))
        {
            /* Set blank map mode */
            opt.blank_map = 1;
        }
        else if (!strcmp(argv[i], "-gbr"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -gbr\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set gaussian blur radius */
            opt.blur_radius = atoi(argv[++i]);
        }
        else if (!strcmp(argv[i], "-aa"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -aa\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set border aa scaling */
            opt.border_blur_scal = atof(argv[++i]);
        }
        else if (!strcmp(argv[i], "-aad"))
        {
            if ((i+1) >= argc)
            {
                printf("Error: missing parameter for -aad\n");
                print_help(argc, argv);
                exit(1);
            }
            /* Set border aa scaling */
            opt.border_blur_diag = atof(argv[++i]);
        }
        else
        {
            /* Print error */
            fprintf(stderr, "Unknown option %s\n", argv[i]);
            print_help(argc, argv);
        }
    }
}

/*
* Perform simple checks on the command-line options.
*/
static void check_options(void)
{
    /* Check map size */
    if (opt.height < 100 || opt.height > 2400)
    {
        /* Error */
        fprintf(stderr, "Map height must be between 100 and 2400.\n");
        exit(1);
    }

    /* Check map width */
    if (opt.width < 100 || opt.width > 3200)
    {
        /* Error */
        fprintf(stderr, "Map width must be between 100 and 3200.\n");
        exit(1);
    }

    /* Check ruggedness */
    if (opt.rugged < 1.0)
    {
        /* Error */
        fprintf(stderr, "Ruggedness must be at least 1.\n");
        exit(1);
    }

    /* Check shadow map ruggedness */
    if (opt.shade_ruggedness < 1.0)
    {
        /* Error */
        fprintf(stderr, "Shadow map ruggedness must be at least 1.\n");
        exit(1);
    }

    /* Check dithering */
    if (opt.terrain_dither < 0.0 || opt.terrain_dither > 1.0)
    {
        /* Error */
        fprintf(stderr, "Dithering must be between 0 and 1.\n");
        exit(1);
    }

    /* Check water line */
    if (opt.water_level < 0 || opt.water_level > 100)
    {
        /* Error */
        fprintf(stderr, "Water level must be between 0 and 100.\n");
        exit(1);
    }

    /* Check tree line */
    if (opt.tree_level < 0 || opt.tree_level > 100)
    {
        /* Error */
        fprintf(stderr, "Tree level must be between 0 and 100.\n");
        exit(1);
    }

    /* Check mountain line */
    if (opt.mountain_level < 0 || opt.mountain_level > 100)
    {
        /* Error */
        fprintf(stderr, "Mountain level must be between 0 and 100.\n");
        exit(1);
    }

    /* Check line ordering */
    if (opt.water_level > opt.tree_level)
    {
        /* Error */
        fprintf(stderr, "Water level must not exceed tree level.\n");
        exit(1);
    }

    /* Check line ordering */
    if (opt.tree_level > opt.mountain_level)
    {
        /* Error */
        fprintf(stderr, "Tree level must not exceed mountain level.\n");
        exit(1);
    }

    /* Check poor line */
    if (opt.poor_level < 0 || opt.poor_level > 100)
    {
        /* Error */
        fprintf(stderr, "Poor level must be between 0 and 100.\n");
        exit(1);
    }

    /* Check rich line */
    if (opt.rich_level < 0 || opt.rich_level > 100)
    {
        /* Error */
        fprintf(stderr, "Rich level must be between 0 and 100.\n");
        exit(1);
    }

    /* Check line ordering */
    if (opt.poor_level > opt.rich_level)
    {
        /* Error */
        fprintf(stderr, "Poor level must not exceed rich level.\n");
        exit(1);
    }

    /* Check number of sea provinces */
    if (opt.num_sea < 0)
    {
        /* Error */
        fprintf(stderr, "Must specify at least 0 sea provinces.\n");
        exit(1);
    }

    /* Check number of land provinces */
    if (opt.num_land < 0)
    {
        /* Error */
        fprintf(stderr, "Must specify at least 0 land provinces.\n");
        exit(1);
    }

    /* Check percentage of special provinces */
    if (opt.amt_special < 0 || opt.amt_special > 100)
    {
        /* Error */
        fprintf(stderr, "Percentage of special provinces must be between 0 and 100.\n");
        exit(1);
    }

    /* Check shore avoidance (sea) */
    if (opt.shore_avoidance_sea < 2)
    {
        /* Error */
        fprintf(stderr, "Sea shore avoidance must be at least 2.\n");
        exit(1);
    }

    /* Check shore avoidance (land) */
    if (opt.shore_avoidance_land < 2)
    {
        /* Error */
        fprintf(stderr, "Land shore avoidance must be at least 2.\n");
        exit(1);
    }

    /* Check spread between capitals */
    if (opt.sea_spread < 2)
    {
        /* Error */
        fprintf(stderr, "Sea spread must be at least 2.\n");
        exit(1);
    }

    /* Check spread between capitals */
    if (opt.land_spread < 2)
    {
        /* Error */
        fprintf(stderr, "Land spread must be at least 2.\n");
        exit(1);
    }

    /* Check sea cost */
    if (opt.sea_cost < 0)
    {
        /* Error */
        fprintf(stderr, "Sea cost must be at least 0.\n");
        exit(1);
    }

    /* Check terrain cost */
    if (opt.terrain_cost < 1)
    {
        /* Error */
        fprintf(stderr, "Terrain cost must be at least 1.\n");
        exit(1);
    }

    /* Check border smoothing */
    if (opt.smooth < 0)
    {
        /* Error */
        fprintf(stderr, "Border smoothing must be at least 0.\n");
        exit(1);
    }

    /* Check min sea neighbours */
    if (opt.min_neighbour_sea < 0)
    {
        /* Error */
        fprintf(stderr, "Minimum sea neighbours must be at least 0.\n");
        exit(1);
    }

    /* Check min land neighbours */
    if (opt.min_neighbour_land < 0)
    {
        /* Error */
        fprintf(stderr, "Minimum land neighbours must be at least 0.\n");
        exit(1);
    }

    /* Check impass */
    if ((opt.impass < 0.0) || (opt.impass > 1.0))
    {
        /* Error */
        fprintf(stderr, "Impass fraction must be between 0.0 and 1.0\n");
        exit(1);
    }

    /* Check sea-sea-impass */
    if ((opt.sea_sea_impass < 0) || (opt.sea_sea_impass > 1))
    {
        /* Error */
        fprintf(stderr, "Sea-sea impassable border setting must be 0 or 1.\n");
        exit(1);
    }

    if ((opt.small_prov_scal < 0.0) || (opt.small_prov_scal > 1.0))
    {
        /* Error */
        fprintf(stderr, "Small province scalar must be between 0.0 and 1.0\n");
        exit(1);
    }

    if (opt.large_prov_scal < 1.0)
    {
        /* Error */
        fprintf(stderr, "Large province scalar must be 1.0 or greater.\n");
        exit(1);
    }

    /* Check impass */
    if ((opt.amt_manysite < 0.0) || (opt.amt_manysite > 1.0))  // we convert from percent to fraction
    {
        /* Error */
        fprintf(stderr, "ManySite province fraction must be between 0.0 and 100.0\n");
        exit(1);
    }
    
    /* Check gaussian blur radius */
    if ((opt.blur_radius < 0) || (opt.blur_radius > 10))
    {
        /* Error */
        fprintf(stderr, "Gaussian blure radius must be in 0-10 range.\n");
        exit(1);
    }
    
    /* amount to weight border color in adj. pixels */
    if (opt.border_blur_scal < 0.0)
    {
        /* Error */
        fprintf(stderr, "Border AA scale value must be >= 0.0.\n");
        exit(1);
    }
    
    /* amount to weight border color in adj. pixels */
    if (opt.border_blur_diag < 0.0)
    {
        /* Error */
        fprintf(stderr, "Border AA diag scale value must be >= 0.0.\n");
        exit(1);
    }
}


/*
* Program begins here.
*/
int main(int argc, char *argv[])
{
    char tga_name[80], map_name[80];
    
    /* Default options */
    set_default_options();

    /* Set options from command line */
    get_command_line_options(argc, argv);

    /* Check options */
    check_options();
    
    /* Seed the random number generator */
    if (opt.seed > 0)
    {
        /* Use command-line seed */
        //srand(opt.seed);
        init_genrand(opt.seed);
    }
    else
    {
        /* Use time */
        opt.seed = (int)time(NULL);

        /* Seed with time value */
        //srand(opt.seed);
        init_genrand(opt.seed);
    }

    /* Generate map array */
    make_map();

    /* Generate terrain */
    generate_map();

    /* Create provinces */
    create_provinces();

    /* Determine tga file and map file names */
    sprintf(tga_name, "%s.tga", opt.map_name);
    sprintf(map_name, "%s.map", opt.map_name);

    /* Generate output file */
    dump_tga(tga_name);

    /* Generate map descriptor file */
    dump_map(map_name, tga_name);

    /* Done */
    return 0;
}
