/*
 * Dominions random map generator.
 *
 * Copyright (c) 2002 Keldon Jones
 *
 * Modified by Philippe Duchon (2004) for compatibility with Dominions 2
 *
 * Modified by Bryon "LintMan" Daly (2004) for assorted improvements and bug fixes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "dommap.h"

/*
* Minimum length needed for a run to be cost effective.
*
* Runs smaller than this will be dumped normally (pixel by pixel).
*/
#define MINIMUM_RUN    2

/* terrain type indexes to terrain color */
long color_map[TOTAL_TERRAIN_TYPES];

int **blur_map;  // map with blurred/smoothed rgb values

/*
* Dump a color value to the targa file.
*/
static void dump_color(FILE *fff, long color)
{
    /* Dump the color byte by byte */
    fputc(color % 256, fff);
    fputc((color >> 8) & 0xff, fff);
    fputc(color >> 16, fff);
}

/*
* Dump a set of raw pixels to the targa file.
*
* Raw pixels have a header byte of (N-1), where N is the count of pixels
* that follow.
*/
static void dump_raw(FILE *fff, long *buffer, int count)
{
    int i;

    /* Dump header byte */
    fputc(count - 1, fff);

    /* Dump each pixel */
    for (i = 0; i < count; i++)
    {
        /* Dump one color */
        dump_color(fff, buffer[i]);
    }
}

/*
* Dump a run of same-colored pixels to the targa file.
*
* This has a header byte of 0x80 + (N-1), where N is the count of same-
* color pixels that follow.
*/
static void dump_run(FILE *fff, long *buffer, int count)
{
    /* Dump header byte */
    fputc(0x80 + count - 1, fff);

    /* Dump the color value */
    dump_color(fff, buffer[0]);
}

/*
* Receive a color value to dump to the targa file.
*
* We now run-length encode (RLE) the file, so we need to collect "runs"
* on color and dump them all at once to save space.
*/
static void add_color(FILE *fff, long color)
{
    static long buffer[128];
    static int current_run, current_color = 0xffffffff;
    static int run_start, buffer_len;
    int i;

    /* Check for end */
    if (color == 0xffffffff)
    {
        /* Flush the buffer */
        if (current_run == buffer_len)
        {
            /* Dump run */
            dump_run(fff, buffer, buffer_len);
        }
        else
        {
            /* Dump raw pixels */
            dump_raw(fff, buffer, buffer_len);
        }

        /* Done */
        return;
    }

    /* Check for match with current run */
    if (color == current_color)
    {
        /* Increment length of run */
        current_run++;

        /* Save color in buffer */
        buffer[buffer_len++] = color;

        /* Check for long enough run */
        if (current_run == MINIMUM_RUN && run_start > 0)
        {
            /* Dump contents of buffer up to run */
            dump_raw(fff, buffer, run_start);

            /* Move run up to beginning of buffer */
            for (i = 0; i < current_run; i++)
            {
                /* Copy color */
                buffer[i] = buffer[i + run_start];
            }

            /* Run starts at beginning of buffer */
            run_start = 0;

            /* Save new buffer length */
            buffer_len = current_run;
        }

        /* Check for full buffer */
        if (buffer_len == 128)
        {
            /* Dump buffer */
            dump_run(fff, buffer, 128);

            /* Clear buffer */
            current_run = 0;
            run_start = 0;
            buffer_len = 0;

            /* Clear current color */
            current_color = 0xffffffff;
        }
    }

    /* No match with current run */
    else
    {
        /* Check for run long enough to dump */
        if (current_run >= MINIMUM_RUN)
        {
            /* Dump run */
            dump_run(fff, buffer, current_run);

            /* Clear buffer */
            current_run = 0;
            run_start = 0;
            buffer_len = 0;
        }

        /* Add color to buffer */
        buffer[buffer_len] = color;

        /* Remember current color */
        current_color = color;

        /* We now have a run of length 1 */
        current_run = 1;
        run_start = buffer_len;

        /* Increase buffer length */
        buffer_len++;

        /* Check for full buffer */
        if (buffer_len == 128)
        {
            /* Dump buffer */
            dump_raw(fff, buffer, 128);

            /* Clear buffer */
            current_run = 0;
            run_start = 0;
            buffer_len = 0;

            /* Clear current color */
            current_color = 0xffffffff;
        }
    }
}

/* Brighten or darken the terrain colour depending on the shadow */
int adjust_for_shadow( long originalColor, double elevation, double shade ) 
{
    int adjustedColor = originalColor;
    long adjustmentFactor = 0;
    long red;
    long green;
    long blue;

    /* Don't shade Capitals or borders, and only shade water if
    opt.do_shade_water is set to true  */
    if( originalColor == opt.capital_color ||
        originalColor == opt.sea_border_color ||
        originalColor == opt.land_border_color ||
        originalColor == opt.cap_border_color ) 
    {
        return originalColor;
    }

    if( originalColor == opt.water_color && !opt.do_shade_water ) 
    {
        return originalColor;
    }
    
    //printf("afs: ele %f   shd %f   mns: %f   mxs: %f   mne: %f   mxe: %f\n", elevation, shade, min_shadow, max_shadow, min_elevation, max_elevation);

    /* Calculate an adjustment from -128 to +128 depending on shadow and elevation */
    if( shade < 0 ) 
    {
        adjustmentFactor = - (long) (opt.shadows_shade_factor * (shade / min_shadow));
    }
    else 
    {
        adjustmentFactor = (long) (opt.shadows_brightness_factor * (shade / max_shadow)); 
    }
    if( elevation < 0 ) 
    {
        adjustmentFactor = adjustmentFactor - (long) (opt.elevation_shade_factor * (elevation / min_elevation));
    }
    else 
    {
        adjustmentFactor = adjustmentFactor + (long) (opt.elevation_brightness_factor * (elevation / max_elevation)); 
    }

    /* Separate the colours into their RGB components */
    red = (originalColor >> 16) & 0xFF;
    green = (originalColor >> 8) & 0xFF;
    blue = originalColor & 0xFF;

    //  fprintf( stderr, "Adjustment factor %d ", adjustmentFactor );
    //  fprintf( stderr, " Original red %d Original Green %d Original Blue %d\n", (int) (red), (int) (green), (int) blue);

    /* Add the adjustment factor to every 8 bit colour in the RGB value */
    blue = blue + adjustmentFactor;
    green = green + adjustmentFactor;
    red = red + adjustmentFactor;

    //  fprintf( stderr, "  Adjusted red %d Adjusted Green %d Adjusted Blue %d\n", (int) (red), (int) (green), (int) blue);

    /* Adjust for overflow or underflow (when one colour component turns
    <0 or >255) */
    if( red < 0 ) { red = 0; }
    if( red > 0xFE ) { red = 0xFE; }
    if( green < 0 ) { green = 0; }
    if( green > 0xFE ) { green = 0xFE; }
    if( blue < 0 ) { blue = 0; }
    if( blue > 0xFE ) { blue = 0xFE; }


    /* Recombine the colours */
    adjustedColor = (red << 16) | (green << 8) | blue;

    red = (adjustedColor >> 16) & 0x000000FF;
    green = (adjustedColor >> 8) & 0x000000FF;
    blue = adjustedColor & 0x000000FF;

    //  fprintf( stderr, " Adjusted red %d Original Green %d Original Blue %d\n", (int) (red), (int) (green), (int) blue);

    //  fprintf( stderr, "Original colour %d Adjusted colour %d\n", originalColor, adjustedColor);
    return adjustedColor;
} // Adjust for shadow





/*
* Dump a targa file containing the map.
*/
void dump_tga(char *name)
{
    FILE *fff;
    int out_height = opt.height+2, out_width = opt.width+2;
    int y, x, map_y, map_x, dir, crow, ccol;
    long color;
    BOOL blur_done = FALSE;
    int prov;

    /* Check verbosity */
    if (opt.verbose)
    {
        /* Message */
        printf("Generating %s file...\n", name);
    }

    // if blank map specified, we just set all land colors to opt.plains_color
    // here, and force terrain type to be plains when .map is generated.  This
    // is both simplier and more likely to create interesting blanks than if we
    // tried to force the map[y][x] fractal terrain generation to be all plains
    // at some earlier point.
    if (opt.blank_map)
    {
        color_map[PLAINS] = opt.plains_color;
        color_map[MOUNTAIN] = opt.plains_color;
        color_map[FARMLAND] = opt.plains_color;
        color_map[FOREST] = opt.plains_color;
        color_map[SWAMP] = opt.plains_color;
        color_map[TUNDRA] = opt.plains_color;
        color_map[WASTELAND] = opt.plains_color;
        // note: don't monkey with border and capital colors! 
    } else 
    {
        color_map[PLAINS] = opt.plains_color;
        color_map[MOUNTAIN] = opt.mountain_color;
        color_map[FARMLAND] = opt.farmland_color;
        color_map[FOREST] = opt.forest_color;
        color_map[SWAMP] = opt.swamp_color;
        color_map[TUNDRA] = opt.tundra_color;
        color_map[WASTELAND] = opt.wasteland_color;
    }
    // These colors are the same, blank or not
    color_map[WATER] = opt.water_color;
    color_map[LAND_BORDER] = opt.land_border_color;
    color_map[SEA_BORDER] = opt.sea_border_color;
    color_map[CAPITAL] = opt.capital_color;
    color_map[CAP_BORDER] = opt.cap_border_color;
    if (opt.cap_border)  // draw special border around capital?
    {
        /* Go through every province (note provs start counting at 1) */
        for (prov = 1; prov <= num_provs; prov++)
        {
            // get location of capital
            crow = provs[prov].cap_y;
            ccol = provs[prov].cap_x;

            // draw border around capital
            for (dir = 0; dir < 8; dir++)  // use dir < 8 for square border
            {
                /* Get new grid location */
                y = crow + dy[dir];
                x = ccol + dx[dir];

                /* Check for out of bounds */
                if ((y < 0) || (y >= opt.height)) continue;
                if ((x < 0) || (x >= opt.width)) continue;

                map[y][x] = CAP_BORDER;

                if (dir < 4)
                {
                y += dy[dir];
                x += dx[dir];

                /* Check for out of bounds */
                if ((y < 0) || (y >= opt.height)) continue;
                if ((x < 0) || (x >= opt.width)) continue;

                map[y][x] = CAP_BORDER;
               }
            }
        }
    }
#ifdef eh
    if (opt.cap_border)  // draw special border around capital?
    {
        /* Go through every province (note provs start counting at 1) */
        for (prov = 1; prov <= num_provs; prov++)
        {
            // get location of capital
            crow = provs[prov].cap_y;
            ccol = provs[prov].cap_x;

            // draw border around capital
            for (dir = 0; dir < 4; dir++)  // use dir < 8 for square border
            {
                /* Get new grid location */
                y = crow + dy[dir];
                x = ccol + dx[dir];

                /* Check for out of bounds */
                if ((y < 0) || (y >= opt.height)) continue;
                if ((x < 0) || (x >= opt.width)) continue;

                map[y][x] = CAP_BORDER;
            }
        }
    }
#endif // eh

    if (opt.blur_radius >= 1)
    {
        printf("blurring terrain...");
        // First, allocate memory for the new blurred map     
        /* Make array of rows */
        blur_map = (int **)calloc(opt.height, sizeof(int *));

        if (blur_map == (void *)NULL)
        {
            printf("Error - memory allocation failed in dump_tga()!  Exiting.\n");
            exit(1);
        }
        /* Make each row */
        for (y = 0; y < opt.height; y++)
        {
            /* Make one row */
            blur_map[y] = (int *)calloc(opt.width, sizeof(int *));
        }

        blur_terrain(); // smooth the terrain to reduce the pixelization
        aa_borders();  // attempt some anti-aliasing along borders
        blur_done = TRUE;
        printf("done\n");
    }

    /* Open the file for writing */
    fff = fopen(name, "wb");

    /* Check for errors */
    if (!fff)
    {
        /* Print error and exit */
        perror("fopen");
        exit(1);
    }

    /* Output a Targa header */
    fputc(0, fff);
    fputc(0, fff);
    fputc(10, fff);
    fputc(0, fff);
    fputc(0, fff);
    fputc(0, fff);
    fputc(0, fff);
    fputc(0, fff);

    /* Output origin coordinates */
    fputc(0, fff);
    fputc(0, fff);
    fputc(0, fff);
    fputc(0, fff);

    /* Output image size */
    fputc(out_width % 256, fff);
    fputc(out_width / 256, fff);
    fputc(out_height % 256, fff);
    fputc(out_height / 256, fff);

    /* 24-bit image */
    fputc(24, fff);

    /* Origin in upper-left corner */
    fputc(32, fff);

    /* Dump the first row (with color key) */
    /* 	for (x = 0; x < out_width; x++) */
    /* 	   { */
    /* 	   /\* Get the color *\/ */
    /* 		if (x < 11) color = color_map[x]; */
    /* 		else color = 0; */

    /* 		/\* Dump the color *\/ */
    /* 		add_color(fff, color); */
    /* 	} */
    
    /* Now dump the map */
    for (y = out_height-1; y >=0; y--)
    {
        for (x = 0; x < out_width; x++)
        {
            /* Compute map location */
            map_y = y + (opt.height - out_height - 1) / 2;
            map_x = x + (opt.width - out_width) / 2;

            /* Check for out of bounds */
            if (map_y < 0 || map_y >= opt.height ||
                map_x < 0 || map_x >= opt.width)
            {
                /* Black */
                color = 0;
            }
            else
            {
                if (!blur_done)
                {
                    /* Get the color of this terrain */
                    color = color_map[map[map_y][map_x]];
                } else
                {
                    /* Get the color of this terrain from blur map */
                    color = blur_map[map_y][map_x];
                }
                if (opt.do_shade) 
                {
                    color = adjust_for_shadow(color, elevation[map_y][map_x], shadow[map_y][map_x]);
                }
            }

            /* Dump the color */
            add_color(fff, color);
        }
    }

    /* Flush the buffer */
    add_color(fff, 0xffffffff);

    /* Done */
    fclose(fff);
}
