package RandomizeMap;
use WeightedSets;

# Flags for various terrain types
$terrainFlag_small = 1;
$terrainFlag_large = 2;
$terrainFlag_sea = 4;
$terrainFlag_river = 8;
$terrainFlag_mountain = 16;
$terrainFlag_swamp = 32;
$terrainFlag_waste = 64;
$terrainFlag_forest = 128;
$terrainFlag_farm = 256;

# A hash of references to all explicitly named weighted sets, keyed by
# name
$new_line = "\n";

$max_units = 1064;
%provinceTerrain = ( );

# All the named keyed by their base name
%matchesByName = ( );

# All defined arrays, keyed by name
%arraysByName = ( );

$randomizationName = "randomized";
$descriptionAddition = " With randomized content added.";

# By default we use the "#land" command rather than "#setland"
$useSetland = 0;

# Parses a #Array definition, and stores the array by name in
# %arraysByName
sub parseArray {
  my $array_description = shift(@_);
  my $name;
  my $expression;
  my %createdHash = ( );

  if( $array_description =~ /^%         # #Array command
                             \s*(\S+)\s*      # A word surrounded by non-space characters
                             =\s*             # An equal sign, possibly surrounded by whitespace
                             {(.*)}/sx ) {     # An expression enclosed in curly brackets.
    $name = $1;
    $expression = $2;
  } # end if
  else {
    print "ERROR. Not a valid array:\n" . $array_description . "\n";
    return;
  } # end else - not a valid array 

  # Add a trailing comma to simplify the regular expression we have to
  # use
  $expression = $expression . ",";

  while( $expression =~ /\s*(\d+)\s*         # A number surrounded by whitespace
	                 =>\s*               # A => operator followed by whitespace
	                 "((\w|\s)+)"\s*           # A string enclosed in ""
	                 ,+/sox ) {     # A comma or the end of the string             
    # Copy this key => value pair into the hash
    $createdHash{ $1 } = $2;
    $expression = $';
  } # end while

  $arraysByName{ $name } = \%createdHash;
} # end sub parseArray

# Called with the name of a configuration file to read
sub read_configuration {
  my $file_name = shift(@_);
  
  open( CONFIGURATION, "<$file_name" );

  while( $line = <CONFIGURATION> ) {
    if( $line =~ /^\s*--/ ) { 
      next;
    } # end if -- comment, skip this line

    if ( $line =~ /^\#RandomizationName\s*=\s*(\w+)/ ) {
      $randomizationName = $1;
      next;
    } # end if -- #RandomizationName

    if ( $line =~ /^\#DescriptionAddition\s*=\s*(.+)$/ ) {
      $descriptionAddition = $1;
      next;
    } # end if -- #DescriptionAddition

    if( $line =~ /^\#UseSetland/ ) {
      $useSetland = 1;
    } # end if -- #UseSetland

    if( $line =~ /^%.*{/ ) {
      # Get all lines, including the end of the array
      my $arrayDescription = $line;

      while( $line !~ /}/) {
	$line = <CONFIGURATION>;
	$arrayDescription = $arrayDescription . $line;
      } # end while -- we still haven't reached the end of the array

      parseArray( $arrayDescription );
    } # end if -- #DefineArray


    if( $line =~ 
        /^\#(\S*)     # Any line starting with a # immediatedly followed
         \s*(=?)\s*  # An optional equal sign, possibly surrounded by whitespace
         (.*)$/x    # Any remaining parameters
      ) {
      $optionName = $1;
      $parameters = $3;

      if( ! $optionName ) {
	next;
      } # end if -- not a configuration line

      if( $2 ) {
	my %currentSet = WeightedSets::parseNamedSetExpression ( $optionName, $parameters );
	# Add the full name to the list of sets with this base-name
	$optionName =~ /^(\w+)(_\w+\(.+\))*$/;
	my $baseName = $1;

	my $listRef = $matchesByName{ $baseName };
	if( $listRef ) {
	  my @nameList = @$listRef;	
	  push @nameList, ($optionName);
	  $matchesByName{ $baseName } = \@nameList;
	}
	else {
	  my @nameList = ( $optionName );
	  $matchesByName{ $baseName } = \@nameList;
	}	
      } # end if -- the line contains an equal sign
      else {
	# Ignore for now
	# $optionValues{ $optionName } = $parameters;
    } # end else -- no equal sign
    } # End if - line matches option    
  } # End while - read the entire file
} # end sub read_configuration


# If there exists a set with this base-name and the exact set of
# modifiers as given in %modifiers, return that set.  If not, find the
# set with this base-name which has the highest number of matches with
# %modifiers and does not have any other modifiers given. If there are
# more than one set which are equally good, return a union of them.
# 
# Basically, the set Name_Mod1(xxx) can match for (Name, (Mod1 => xxx,
# Mod2 => yyy)) but not for (Name, ()) which is more generic or (Name,
# (Mod1 => yyy)) which is incompatible.
sub getMatchingSet {
  my ($baseName, %modifiers ) = @_;
  my $fullName = $baseName;
  my $i;
  
  my $possibleMatchesRef = $matchesByName{ $baseName };
  my $currentMaxMatches = 0;
  my @maxMatchesList = ( );

  MATCHES: foreach my $possibleMatch (@$possibleMatchesRef) {
    my $matches = 0;

    # Make sure this match does not have any incomaptible modifiers
    $possibleMatch =~ /^([a-zA-Z0-9]+)(_.*)/; 

    my $remainder = $2;

    # How many matching modifiers does it have?
    while( $remainder ) {
      $remainder =~ 
	/^_([a-zA-Z0-9]+) # An underscore followed by an alphanumeric word
	\(([a-zA-Z0-9{}<>]+)\)         # An alphanumeric word enclosed in parantheses
	(.*)
	/x;

      my $modifierName = $1;
      my $modifierValue = $2;

      $remainder = $3;
      my $hashedValue = $modifiers{ $modifierName };
      if( $modifierValue =~ /^\d+/ ) {
	if( $hashedValue eq $modifierValue ) {
	  # We have a matching modifier
	  $matches = $matches + 1;
	}
	else {
	  next MATCHES;
	}
      }
      # If this modifier is a set description, or the name of a set
      else {
	my %set = ( );
	if(  $modifierValue =~ /\{.*\}/ ) {
	  %set = WeightedSets::parseSet( $modifierValue );
	}
	elsif ( $modifierValue =~/(\d+)/ ) {
	  %set = ( $1 );
	}
	else {
	  %set = WeightedSets::getNamedSet( $modifierValue );
	}

	if( $set{ $hashedValue } ) {
	  $matches = $matches + 1;
	}
	else {
	  # $possibleMatch either contains a modifier which is not in
	  # %modifiers, or it has the wrong value for one of the
	  # modifiers in %modifiers
	  next MATCHES;
	} # end else
      } # end else
    } # end while
    # Is this more than the current maximum?
    if( $matches > $currentMaxMatches ) {
      # If so, clear the @maxMatchesList, and push the current match
      # into it and set the new maximum?
      $currentMaxMatches = $matches;
      @maxMatchesList = ( );
    } # end if 

    # Is it equal to the current maximum?
    if( $matches == $currentMaxMatches ) {
      # If so, add it to @maxMatchesList
      push @maxMatchesList, ( $possibleMatch );
    } # end if
  } # end for -- every possible match

  # Return the union of all the sets in @maxMatchesList
  my $firstName = pop @maxMatchesList;
  my %resultSet = WeightedSets::getNamedSet( $firstName );

  while( my $nextName = pop @maxMatchesList ) {
    my %nextSet = WeightedSets::getNamedSet( $nextName );
    %resultSet = WeightedSets::setUnion( \%resultSet, \%nextSet );
  } # end while

  return %resultSet;
} # end sub getMatchingSet

sub clear_configuration {
  %provinceTerrain = ( );
  %matchesByName = ( );
  $randomizationName = "randomized";
  $descriptionAddition = " With randomized content added.";

  WeightedSets::clearSets;
} # end sub clear_configuration

sub randomize_map {
  $new_line = "\n";
  my ($original_map_file, $configuration_file) = @_;
  my $province_count = 0;

  clear_configuration;
  read_configuration( $configuration_file );

  $original_map_file =~ /^(.*)\.map$/;
  my $randomized_map_file = $1 . "_" . $randomizationName . ".map";

  open( ORIGINAL, "<$original_map_file" );
  open( RANDOMIZED, ">$randomized_map_file" );

  # Copy the terrain and neighbor fields
  while( $line = <ORIGINAL> ) {
    if( $line =~ /^\#terrain\s+(\d+)\s+(\d+)/ ) {      
      $province_count++;
      $provinceTerrain{ $1 } = $2;
      print RANDOMIZED $line;
    } elsif( $line =~ /^\#description\s\"(.+)\".*$/ ) {
      print RANDOMIZED "\#description \"" . $1 . " " . $descriptionAddition . "\"" . $new_line;
    } elsif( $line =~ /^\#dom2title\s+(\w+)/ ) {
      print RANDOMIZED "\#dom2title " . $1 . "_" . $randomizationName . $new_line;
    } elsif( $line =~ /^\#neighbo[u]r/ ||
	     $line =~ /^\#imagefile/ ||
	     $line =~ /^\#domversion/ ||
	     $line =~ /^\#defaultzoom/ ||
	     $line =~ /^\#nostart/ || 
	     $line =~ /^\s*$/)   # Empty lines
	     {
	       # Copy all lines that matches the above
	print RANDOMIZED $line;
      } # end if
    } # end while
    
    print RANDOMIZED $new_line;
    
    # Set up allies
    for( $i = 0; $i < 13; ++$i ) {
      $first_ally = int( 18 * rand( ) );
      while ( ($second_ally = int( 18 * rand( ) )) == $first_ally ) {
	# Do nothing
      } # End while
      
      print RANDOMIZED "#allies $first_ally $second_ally" . $new_line;  
    } # end for
    
    print RANDOMIZED $new_line;

  print( "Province Count: $province_count" . $new_line );

    for( $i = 1; $i <= $province_count; ++$i ) {
      my %provinceModifiers = ( );

      # ===> Province number <===
      if( $useSetland ) {
	print RANDOMIZED $new_line . "#setland $i" . $new_line;
      }
      else {
	print RANDOMIZED $new_line . "#land $i" . $new_line;
      }

      # ===> Find the Terrain for this province <===
      my $terrain = $provinceTerrain{ $i };
      $provinceModifiers{ "Terrain" } = $terrain;

      if( $terrain & $terrainFlag_small ) {
	$provinceModifiers{ "TerrainSmall" } = 1;
      }
      if( $terrain & $terrainFlag_large ) {
	$provinceModifiers{ "TerrainLarge" } = 1;
      }
      if( $terrain & $terrainFlag_sea ) {
	$provinceModifiers{ "TerrainSea" } = 1;
	$provinceModifiers{ "TerrainLand" } = 0;
      }
      else {
	$provinceModifiers{ "TerrainSea" } = 0;
	$provinceModifiers{ "TerrainLand" } = 1;
      }
      if( $terrain & $terrainFlag_river ) {
	$provinceModifiers{ "TerrainRiver" } = 1;
      }
      if( $terrain & $terrainFlag_mountain ) {
	$provinceModifiers{ "TerrainMountain" } = 1;
      }
      if( $terrain & $terrainFlag_swamp ) {
	$provinceModifiers{ "TerrainSwamp" } = 1;
      }
      if( $terrain & $terrainFlag_waste ) {
	$provinceModifiers{ "TerrainWaste" } = 1;
      }
      if( $terrain & $terrainFlag_forest ) {
	$provinceModifiers{ "TerrainForest" } = 1;
      }
      if( $terrain & $terrainFlag_farm ) {
	$provinceModifiers{ "TerrainFarm" } = 1;
      }

      # ===> Select Population Type <===
      my %selectionSet = getMatchingSet( "PopulationType", %provinceModifiers );
      my $populationType = 
	WeightedSets::chooseRandomElement( %selectionSet );
      $provinceModifiers{ "PopulationType" } = $populationType;
      if( $populationType ) {
	print RANDOMIZED "#poptype $populationType" . $new_line;
      }
      
      # ===> Select fort <===
      %selectionSet = getMatchingSet( "Fort", %provinceModifiers );
      my $fort = 
	WeightedSets::chooseRandomElement( %selectionSet );
      $provinceModifiers{ "Fort" } = $fort;
      if( $fort ) {
	print RANDOMIZED "#fort $fort" . $new_line;
      } # End if -- there is a fort

      # ===> Select lab or not <===
      %selectionSet = getMatchingSet( "Lab", %provinceModifiers );
      my $lab = 
	WeightedSets::chooseRandomElement( %selectionSet );
      $provinceModifiers{ "Lab" } = $lab;
      if( $lab ) {
	print RANDOMIZED "#lab" . $new_line;
      } # End if -- there is a fort

      # ===> Select temple or not <===
      %selectionSet = getMatchingSet( "Temple", %provinceModifiers );
      my $temple = 
	WeightedSets::chooseRandomElement( %selectionSet );
      $provinceModifiers{ "Temple" } = $temple;
      if( $temple ) {
	print RANDOMIZED "#temple" . $new_line;
      } # End if -- there is a fort

      # =====================
      # ===> Magic Sites <===
      # =====================
      $mapRef = $arraysByName{ "MagicSiteNames" };
      if( $mapRef ) {
	%mapSet = %$mapRef;

	# ===> Number of Known Magic Sites <===
	%selectionSet = getMatchingSet( "KnownMagicSiteNumber", %provinceModifiers );
	my $knownChecks = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	$provinceModifiers{ "KnownMagicSiteNumber" } = $knownChecks;

	# ===> Select known magic sites <===
	for( $siteCount = 0; $siteCount < $knownChecks; ++$siteCount ) {
	  %selectionSet = getMatchingSet( "KnownMagicSite", %provinceModifiers );
	  my $magicSiteNumber = 
	    WeightedSets::chooseRandomElement( %selectionSet );
	  my $magicSiteName = $mapSet{ $magicSiteNumber };	
	  $provinceModifiers{ "KnownMagicSite" } = $magicSiteNumber;
	  if( $magicSiteName ) {
	    print RANDOMIZED "#knownfeature \"$magicSiteName\"" . $new_line;
	  }
	} # end for

	# ===> Number of Unknown Magic Sites <===
	%selectionSet = getMatchingSet( "UnknownMagicSiteNumber", %provinceModifiers );
	my $unknownChecks = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	$provinceModifiers{ "UnknownMagicSiteNumber" } = $unknownChecks;

	for( $siteCount = 0; $siteCount < $unknownChecks; ++$siteCount ) {
	  %selectionSet = getMatchingSet( "UnknownMagicSite", %provinceModifiers );
	  $magicSiteNumber = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	  $magicSiteName = $mapSet{ $magicSiteNumber };	
	  $provinceModifiers{ "UnknownMagicSite" } = $magicSiteNumber;
	  if( $magicSiteName ) {
	    print RANDOMIZED "#feature \"$magicSiteName\"" . $new_line;
	  }
	} # end for
      } # End if -- Magic sites have been specified

      # ====================
      # ===> Commanders <===
      # ====================

      # ===> Select Number of Commanders <===
      %selectionSet = getMatchingSet( "CommanderNumber", %provinceModifiers );
      my $commanderNumber = 
	WeightedSets::chooseRandomElement( %selectionSet );
      $provinceModifiers{ "CommanderNumber" } = $commanderNumber;

      # ===> Create Commanders <===
      for( my $commanderCount = 0; $commanderCount < $commanderNumber; $commanderCount++ ) {

	# ===> Select Commander Type <===
	%selectionSet = getMatchingSet( "CommanderType", %provinceModifiers );
	my $commanderType = 
	  WeightedSets::chooseRandomElement( %selectionSet );

	if( ! $commanderType ) { next; }
	$provinceModifiers{ "CommanderType" } = $commanderType;
	print RANDOMIZED "#commander $commanderType" . $new_line;

	# ===> Select magic items <===
	$mapRef = $arraysByName{ "MagicItemNames" };
	if( $mapRef ) {
	  %mapSet = %$mapRef;
	  %selectionSet = getMatchingSet( "MagicItemNumber", %provinceModifiers );
	  my $numberOfItems = 
	    WeightedSets::chooseRandomElement( %selectionSet );
	  $provinceModifiers{ "MagicItemNumber" } = $numberOfItems;
	  for( $itemCount = 0; $itemCount < $numberOfItems; ++$itemCount ) {
	    %selectionSet = getMatchingSet( "MagicItem", %provinceModifiers );
	    my $itemNumber = 
	      WeightedSets::chooseRandomElement( %selectionSet );
	    $provinceModifiers{ "MagicItem" } = $itemNumber;
	    my $itemName = $mapSet{ $itemNumber };
	    print RANDOMIZED "#additem \"$itemName\"" . $new_line;
	  } # end for -- Every Magic item
	} # end if -- There is an array of item names

	# ===> Select RandomEquip <===
	%selectionSet = getMatchingSet( "RandomEquip", %provinceModifiers );
	my $randomEquip = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	if( $randomEquip ) {
	  $provinceModifiers{ "RandomEquip" } = $randomEquip;
	  print RANDOMIZED "#randomequip $randomEquip" . $new_line;
	}

	# ===> Select Bodyguard Type <===
	%selectionSet = getMatchingSet( "BodyguardType", %provinceModifiers );
	my $bodyguardType = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	$provinceModifiers{ "BodyguardType" } = $bodyguardType;

	# ===> Select Bodyguard Size <===
	%selectionSet = getMatchingSet( "BodyguardSize", %provinceModifiers );
	my $bodyguardSize = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	$provinceModifiers{ "BodyguardSize" } = $bodyguardSize;
	if( $bodyguardType && $bodyguardSize ) {
	  print RANDOMIZED "#bodyguards $bodyguardSize $bodyguardType" . $new_line;
	} # end if 

	# ===> Units Number <===
	%selectionSet = getMatchingSet( "UnitsNumber", %provinceModifiers );
	my $unitsNumber = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	$provinceModifiers{ "UnitsNumber" } = $unitsNumber;

	for( my $unitsCount = 0; $unitsCount < $unitsNumber; $unitsCount++ ) {	
	  # ===> Unit Type <===
	  %selectionSet = getMatchingSet( "UnitsType", %provinceModifiers );
	  my $unitsType = 
	  WeightedSets::chooseRandomElement( %selectionSet );
	  $provinceModifiers{ "UnitsType" } = $unitsType;

	  # ===> Unit Size <===
	  %selectionSet = getMatchingSet( "UnitsSize", %provinceModifiers );
	  my $unitsSize = 
	    WeightedSets::chooseRandomElement( %selectionSet );
	  $provinceModifiers{ "UnitsSize" } = $unitsSize;

	  if( $unitsType && $unitsSize ) {
	    print RANDOMIZED "#units $unitsSize $unitsType" . $new_line;
	  }
	} # End - make all units for this commander
      } # End - Make all commanders
    } # End - Make all provinces
} # end sub randomize_map

1;
