using System;
using System.Collections;
using System.Data;
using System.IO;
using System.Net;
using System.Text;

using Microsoft.DirectX;
using Microsoft.DirectX.DirectPlay;

namespace LOLGameServer
{
    public class ServerGameInstance
    {
        #region Static Functions
        public static ServerGameInstance GetGameInstanceForAttachedClient(AttachedClient attachedClient) 
        {
			try 
			{
				lock (Base_GameServer.GameInstances) 
				{
					foreach (ServerGameInstance gameInstance in Base_GameServer.GameInstances.Values) 
					{
						if (gameInstance.IsPlayerInGame(attachedClient.Username)) 
						{
							return gameInstance;
						}
					}
				}
			} 
			catch {}
            return null;
        }

        #endregion
        
        #region Private Member Variables
        private Hashtable _attachedClients=null;
        private int _serverGameInstanceID=0;
        private string _turnInProgress="";
        private string _lastEndTurnState;
        private string _challengerUsername="";
        private int _challengerUserID=0;
        private string _challengeeUsername="";
        private int _challengeeUserID=0;
        private Hashtable _stateCache;

        private bool _gameHasStarted=false;
        private bool _gameHasEnded=false;
		
        private bool _challengeeDeclaredEndTurn=false;
        private bool _challengerDeclaredEndTurn=false;        

		//these are set later by the clients when they log in, because only the client knows if it's a demo client or full client.
		//consider these SEPERATE from the rest of the data.
		private bool _challengeeIsRegistered=true;
		private bool _challengerIsRegistered=true;

		#endregion

        #region Accessor Functions
        public int GameID 
        {
            get {return _serverGameInstanceID;}
        }
        public bool GameHasStarted 
        {
            get {return _gameHasStarted;}
            set {_gameHasStarted=value;}
        }
        public bool GameHasEnded 
        {
            get {return _gameHasEnded;}
            set {_gameHasEnded=value;}
        }
        public string ChallengerUsername 
        {
            get {return _challengerUsername;}
        }
        public int ChallengerID 
        {
            get {return _challengerUserID;}
        }
        public string ChallengeeUsername 
        {
            get {return _challengeeUsername;}
        }
        public int ChallengeeID 
        {
            get {return _challengeeUserID;}
        }
        public Hashtable AttachedClients
        {
            get {return _attachedClients;}
        }
		public bool ChallengeeIsRegistered 
		{
			get {return _challengeeIsRegistered;}
			set {_challengeeIsRegistered=value;}
		}
		public bool ChallengerIsRegistered 
		{
			get {return _challengerIsRegistered;}
			set {_challengerIsRegistered=value;}
		}

		#endregion

        #region Constructor
        public ServerGameInstance(int serverGameInstanceID, string challengerUsername, int challengerUserID, string challengeeUsername, int challengeeUserID)
        {
            _attachedClients=new Hashtable();
            
            _challengerUsername=challengerUsername;
            _challengerUserID=challengerUserID;
            _challengeeUsername=challengeeUsername;
            _challengeeUserID=challengeeUserID;
            
            _stateCache=new Hashtable();
            _serverGameInstanceID=serverGameInstanceID;

        }
        #endregion

        #region Public Methods
        public bool IsPlayerInGame(string username) 
        {
            bool __isPlayerAttached = false;
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (__attachedClient.Username.ToLower() == username.ToLower()) 
                    {
                        __isPlayerAttached  =true;
                        break;
                    }
                }
            }
            return __isPlayerAttached;
        }

        public bool IsPlayerAbleToJoinGame(string username, ClientRole clientRole) 
        {
            bool __isPlayerAlreadyAttached = false;
            bool __playerCanFillRole= false;

            //is the player already in the game?
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (__attachedClient.Username.ToLower() == username.ToLower()) 
                    {
                        __isPlayerAlreadyAttached =true;
                        break;
                    }
                }
            }

            //is the connecting player 
            if (username.ToLower() == _challengerUsername.ToLower() && clientRole==ClientRole.Challenger) 
            {
                __playerCanFillRole=true;
            }
            if (username.ToLower() == _challengeeUsername.ToLower() && clientRole==ClientRole.Challengee) 
            {
                __playerCanFillRole=true;
            }
            if (clientRole==ClientRole.Observer) 
            {
                __playerCanFillRole=true;
            }

            if (__playerCanFillRole==true && __isPlayerAlreadyAttached==false) 
            {
                return true;
            } 
            else 
            {
                return false;
            }
        }
        public bool IsChallengerConnected() 
        {
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients) 
                {
                    if (__attachedClient.Username.ToLower() == _challengerUsername.ToLower()) 
                    {
                        return true;
                    }
                }
            }
            return false;
        }
        public bool IsChallengeeConnected() 
        {
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients) 
                {
                    if (__attachedClient.Username.ToLower() == _challengeeUsername.ToLower()) 
                    {
                        return true;
                    }
                }
            }

            return false;
        }
        public void AddOrUpdateStateCache(string key, string cacheValue) 
        {
            if (key=="") 
            {
                return;
            }
            _stateCache[key]=cacheValue;
        }

        public void RemoveStateCache(string key) 
        {
            if (!_stateCache.ContainsKey(key)) 
            {
                //there is no cache for that key.  Send "" and expect the client to deal w/ it.
                return;
            }
            _stateCache.Remove(key);
        }

        public string GetStateCacheFromKey(string key) 
        {
            if (!_stateCache.ContainsKey(key)) 
            {
                //there is no cache for that key.  Send "" and expect the client to deal w/ it.
                return "";
            }

            // there is a value -- let's return it.
            return (string) _stateCache[key];
        }

        public void AddPlayerToGame(string username, AttachedClient client) 
        {
            _attachedClients.Add(username, client);
        }
        public void RemovePlayerFromGame(string username) 
        {
            _attachedClients.Remove(username);

            //check to see if there's any players still attached to this, and if not, let's get rid of it.
            if (_attachedClients.Count==0) 
            {
                Base_GameServer.GameInstances.Remove(this.GameID);
                //LogMessage("A game (ID# " + this.GameID + ") has been flushed from memory since all players have left.");
            }
        }
        
        public void SendGameActionToAllOtherPlayersInGame(AttachedClient attachedClient, int gameMessageNumber, string key, string gameActionString, string gameStateConfirmationString, Server gameServer) 
        {
            NetworkPacket __returnPacket = new NetworkPacket();
            __returnPacket.Write(GameMessageType.GameAction);
            __returnPacket.Write(gameMessageNumber);
            __returnPacket.Write(key);
            __returnPacket.Write(gameActionString);
            __returnPacket.Write(gameStateConfirmationString);
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (attachedClient.Username.ToLower() != __attachedClient.Username.ToLower()) 
                    {
                        //send the state string to all other players.  
                        gameServer.SendTo(__attachedClient.PlayerID, __returnPacket, Base_GameServer.PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
                    }
                }
            }
            __returnPacket.Dispose();
        }
        public void SendStateStringToAllOtherPlayersInGame(AttachedClient attachedClient, string key, string stateString, Server gameServer) 
        {
            NetworkPacket __returnPacket = new NetworkPacket();
            __returnPacket.Write(GameMessageType.SendStateString);
            __returnPacket.Write(key);
            __returnPacket.Write(stateString);
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (attachedClient.Username.ToLower() != __attachedClient.Username.ToLower()) 
                    {
                        //send the state string to all other players.  
                        gameServer.SendTo(__attachedClient.PlayerID, __returnPacket, Base_GameServer.PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
                    }
                }
            }
            __returnPacket.Dispose();
        }
        public void SendAllStateStringsToNewPlayer(AttachedClient attachedClient, Server gameServer)
        {
            lock (_stateCache) 
            {
                foreach (string __stateKey in _stateCache.Keys)
                {
                    NetworkPacket __returnPacket = new NetworkPacket();
                    __returnPacket.Write(GameMessageType.SendStateString);
                    __returnPacket.Write(__stateKey);
                    __returnPacket.Write((string) _stateCache[__stateKey]);
                    gameServer.SendTo(attachedClient.PlayerID, __returnPacket, Base_GameServer.PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
                    __returnPacket.Dispose();
                }
            }
        }
        public void SendPacketToPlayer(AttachedClient attachedClient, NetworkPacket __packetToSend, Server gameServer)
        {
            lock (_attachedClients) 
            {
                //send the state string to the designated player.  
                gameServer.SendTo(attachedClient.PlayerID, __packetToSend, Base_GameServer.PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
            }
        }
        public void SendPacketToAllConnectedPlayersInGame(NetworkPacket __packetToSend, Server gameServer)
        {
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    //send the state string to all other players.  
                    try
                    {
                        gameServer.SendTo(__attachedClient.PlayerID, __packetToSend, Base_GameServer.PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
                    } 
                    catch (Exception e) 
                    {
                        //LogMessage("ERROR:  We tried to send a message to players, but we couldn't find that AttachedClient.");
                    }
                }
            }
        }
        public void SendPacketToAllOtherConnectedPlayersInGame(AttachedClient attachedClient, NetworkPacket __packetToSend, Server gameServer)
        {
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (attachedClient.Username.ToLower() != __attachedClient.Username.ToLower()) 
                    {
                        //send the state string to all other players.  
                        try 
                        {
                            gameServer.SendTo(__attachedClient.PlayerID, __packetToSend, Base_GameServer.PacketTimoutDuration, SendFlags.Guaranteed | SendFlags.NoLoopback);
                        }
                        catch  (Exception e) 
                        {
                            //LogMessage("ERROR:  We tried to send a message to players, but we couldn't find that AttachedClient.");
                        }
                    }
                }
            }
        }
        public void PlayerDeclaredEndTurn(int playerID)
        {
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (__attachedClient.UserID==playerID) 
                    {
                        if (__attachedClient.ClientRole==ClientRole.Challenger)
                        {
                            _challengerDeclaredEndTurn=true;
                        }
                        else if (__attachedClient.ClientRole==ClientRole.Challenger)
                        {
                            _challengeeDeclaredEndTurn=true;
                        }
                        else 
                        {
                            //TROUBLE!  We got an ENDTURN message from someone other than the challenger/challengee in the game!
                        }
                        return;
                    }
                }            
            }
        }
        public bool OtherPlayerConfirmedEndTurn (int playerID)
        {
            //NOTE:  if this ever return anything other than false, then we should ABORT ABORT ABORT!  (hackery or buggery is afoot.)
            bool __foundOtherPlayer=false;
            lock (_attachedClients) 
            {
                foreach (AttachedClient __attachedClient in _attachedClients.Values) 
                {
                    if (__attachedClient.UserID==playerID) 
                    {
                        if (__attachedClient.ClientRole==ClientRole.Challenger && _challengeeDeclaredEndTurn)
                        {
                            __foundOtherPlayer=true;
                        }
                        else if (__attachedClient.ClientRole==ClientRole.Challenger && _challengerDeclaredEndTurn)
                        {
                            __foundOtherPlayer=true;
                        }
                        else 
                        {
                            //TROUBLE!  We got an ENDTURN message from someone other than the challenger/challengee in the game!
                        }
                    }
                }
            }
            //reset both for next time.
            _challengerDeclaredEndTurn=false;
            _challengeeDeclaredEndTurn=false;
            return __foundOtherPlayer;
        }
        public void NukePlayer(AttachedClient attachedClient, bool notifyOtherPlayers, Server gameServer) 
        {
            NetworkPacket __returnPacket;

            //nuke from any server game instance it belongs to
            //if (ServerGameInstance.GetGameInstanceForAttachedClient(attachedClient)!=null) 
            //{
                //ServerGameInstance __gameInstance = ServerGameInstance.GetGameInstanceForAttachedClient(attachedClient);
            if (notifyOtherPlayers) 
            {
                //tell the other attached players that a player left.
                __returnPacket = new NetworkPacket();
                __returnPacket.Write(GameMessageType.PlayerLoggedOff);
                __returnPacket.Write(attachedClient.Username);
                SendPacketToAllOtherConnectedPlayersInGame(attachedClient,__returnPacket, gameServer);
                __returnPacket.Dispose();
            }
            RemovePlayerFromGame(attachedClient.Username);
                //__gameInstance.RemovePlayerFromGame(attachedClient.Username);
            //}

	        //remove this client from this game instance
            //AttachedClients.Remove(attachedClient.Username);
            //remove this client from the overall logged-in client pool
            Base_GameServer.AttachedClients.Remove(attachedClient.Username);

            try 
            {
                gameServer.DestroyClient(attachedClient.PlayerID,new NetworkPacket());
            } 
            catch 
            {
                Base_GameServer.BaseGameServer.LogMessage("ERROR:  We tried to destroy a client (PlayerID=" + attachedClient.PlayerID.ToString() + ", but we encountered an error.");
            }
        }
        #endregion
    }
}
